---
name: nodejs-performance
description: Node.js performance optimization techniques including async patterns, memory management, clustering, caching strategies, and profiling tools.
---

# Node.js Performance Optimization

Node.js performance optimization techniques including async patterns, memory management, clustering, caching strategies, and profiling tools.

## Async Patterns

### Avoid Blocking the Event Loop

```typescript
// BAD - Blocking
function processLargeArray(items: number[]): number {
  let sum = 0;
  for (const item of items) {
    sum += heavyComputation(item);
  }
  return sum;
}

// GOOD - Chunked processing
async function processLargeArrayAsync(items: number[]): Promise<number> {
  const CHUNK_SIZE = 1000;
  let sum = 0;

  for (let i = 0; i < items.length; i += CHUNK_SIZE) {
    const chunk = items.slice(i, i + CHUNK_SIZE);
    sum += chunk.reduce((acc, item) => acc + heavyComputation(item), 0);

    // Yield to event loop
    await new Promise(resolve => setImmediate(resolve));
  }

  return sum;
}

// GOOD - Use Worker Threads for CPU-intensive work
import { Worker, isMainThread, parentPort, workerData } from 'worker_threads';

if (isMainThread) {
  async function runInWorker(data: number[]): Promise<number> {
    return new Promise((resolve, reject) => {
      const worker = new Worker(__filename, { workerData: data });
      worker.on('message', resolve);
      worker.on('error', reject);
    });
  }
} else {
  const result = processLargeArray(workerData);
  parentPort?.postMessage(result);
}
```

### Parallel vs Sequential

```typescript
// Sequential - Slow
async function fetchAllSequential(urls: string[]) {
  const results = [];
  for (const url of urls) {
    results.push(await fetch(url));
  }
  return results;
}

// Parallel - Fast
async function fetchAllParallel(urls: string[]) {
  return Promise.all(urls.map(url => fetch(url)));
}

// Parallel with concurrency limit
async function fetchWithLimit(urls: string[], limit: number) {
  const results: Response[] = [];

  for (let i = 0; i < urls.length; i += limit) {
    const chunk = urls.slice(i, i + limit);
    const chunkResults = await Promise.all(chunk.map(url => fetch(url)));
    results.push(...chunkResults);
  }

  return results;
}

// Using p-limit for fine-grained control
import pLimit from 'p-limit';

const limit = pLimit(5); // Max 5 concurrent

async function fetchWithPLimit(urls: string[]) {
  return Promise.all(
    urls.map(url => limit(() => fetch(url)))
  );
}
```

## Memory Management

### Stream Large Files

```typescript
import { createReadStream, createWriteStream } from 'fs';
import { pipeline } from 'stream/promises';
import { Transform } from 'stream';

// BAD - Loads entire file into memory
async function processFileBad(path: string) {
  const content = await fs.promises.readFile(path, 'utf-8');
  const processed = processContent(content);
  await fs.promises.writeFile('output.txt', processed);
}

// GOOD - Stream processing
async function processFileGood(inputPath: string, outputPath: string) {
  const transform = new Transform({
    transform(chunk, encoding, callback) {
      const processed = processChunk(chunk.toString());
      callback(null, processed);
    },
  });

  await pipeline(
    createReadStream(inputPath),
    transform,
    createWriteStream(outputPath)
  );
}

// JSON streaming for large arrays
import { parser } from 'stream-json';
import { streamArray } from 'stream-json/streamers/StreamArray';

async function processLargeJSON(path: string) {
  const pipeline = createReadStream(path)
    .pipe(parser())
    .pipe(streamArray());

  for await (const { value } of pipeline) {
    await processItem(value);
  }
}
```

### Memory Leak Prevention

```typescript
// BAD - Memory leak with closures
function createHandler() {
  const largeData = new Array(1000000).fill('data');

  return function handler(req, res) {
    // largeData is retained even if not used
    res.send('OK');
  };
}

// GOOD - Don't capture unnecessary data
function createHandler() {
  return function handler(req, res) {
    res.send('OK');
  };
}

// BAD - Growing event listeners
function addListeners() {
  emitter.on('data', handleData);
}
// Called multiple times = memory leak

// GOOD - Remove listeners
function addListeners() {
  emitter.on('data', handleData);
  return () => emitter.off('data', handleData);
}

// GOOD - Use once for one-time listeners
emitter.once('data', handleData);
```

## Clustering

```typescript
import cluster from 'cluster';
import os from 'os';
import express from 'express';

const numCPUs = os.cpus().length;

if (cluster.isPrimary) {
  console.log(`Primary ${process.pid} is running`);

  // Fork workers
  for (let i = 0; i < numCPUs; i++) {
    cluster.fork();
  }

  cluster.on('exit', (worker, code, signal) => {
    console.log(`Worker ${worker.process.pid} died`);
    // Restart worker
    cluster.fork();
  });
} else {
  const app = express();

  app.get('/', (req, res) => {
    res.send(`Worker ${process.pid}`);
  });

  app.listen(3000);
  console.log(`Worker ${process.pid} started`);
}

// Or use PM2 for production
// pm2 start app.js -i max
```

## Caching Strategies

### In-Memory Cache

```typescript
import NodeCache from 'node-cache';

const cache = new NodeCache({
  stdTTL: 300, // 5 minutes default
  checkperiod: 60,
  maxKeys: 1000,
});

async function getCachedUser(id: string): Promise<User> {
  const cached = cache.get<User>(`user:${id}`);
  if (cached) return cached;

  const user = await db.users.findById(id);
  cache.set(`user:${id}`, user, 600); // 10 minutes
  return user;
}

// Cache decorator
function cached(ttl: number) {
  return function (target: any, key: string, descriptor: PropertyDescriptor) {
    const original = descriptor.value;

    descriptor.value = async function (...args: any[]) {
      const cacheKey = `${key}:${JSON.stringify(args)}`;
      const cached = cache.get(cacheKey);
      if (cached) return cached;

      const result = await original.apply(this, args);
      cache.set(cacheKey, result, ttl);
      return result;
    };

    return descriptor;
  };
}
```

### Redis Cache

```typescript
import Redis from 'ioredis';

const redis = new Redis(process.env.REDIS_URL);

async function getCachedData<T>(
  key: string,
  fetcher: () => Promise<T>,
  ttl: number = 300
): Promise<T> {
  const cached = await redis.get(key);
  if (cached) {
    return JSON.parse(cached);
  }

  const data = await fetcher();
  await redis.setex(key, ttl, JSON.stringify(data));
  return data;
}

// Cache invalidation
async function invalidatePattern(pattern: string) {
  const keys = await redis.keys(pattern);
  if (keys.length > 0) {
    await redis.del(...keys);
  }
}

// Usage
const user = await getCachedData(
  `user:${id}`,
  () => db.users.findById(id),
  600
);

// Invalidate on update
await db.users.update(id, data);
await redis.del(`user:${id}`);
```

## Database Optimization

### Connection Pooling

```typescript
import { Pool } from 'pg';

const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
  max: 20, // Max connections
  idleTimeoutMillis: 30000,
  connectionTimeoutMillis: 2000,
});

// Use pool for queries
async function query(text: string, params?: any[]) {
  const client = await pool.connect();
  try {
    return await client.query(text, params);
  } finally {
    client.release();
  }
}
```

### Query Optimization

```typescript
// BAD - N+1 query problem
async function getPostsWithAuthors() {
  const posts = await db.posts.findAll();
  for (const post of posts) {
    post.author = await db.users.findById(post.authorId);
  }
  return posts;
}

// GOOD - Single query with join
async function getPostsWithAuthors() {
  return db.posts.findAll({
    include: [{ model: User, as: 'author' }],
  });
}

// GOOD - Batch loading
import DataLoader from 'dataloader';

const userLoader = new DataLoader(async (ids: string[]) => {
  const users = await db.users.findAll({
    where: { id: ids },
  });
  const userMap = new Map(users.map(u => [u.id, u]));
  return ids.map(id => userMap.get(id));
});

async function getPostsWithAuthors() {
  const posts = await db.posts.findAll();
  await Promise.all(
    posts.map(async post => {
      post.author = await userLoader.load(post.authorId);
    })
  );
  return posts;
}
```

## Profiling

### CPU Profiling

```bash
# Start with profiling
node --prof app.js

# Process the log
node --prof-process isolate-*.log > profile.txt

# Or use clinic.js
npx clinic doctor -- node app.js
npx clinic flame -- node app.js
```

### Memory Profiling

```typescript
// Check memory usage
function logMemory() {
  const used = process.memoryUsage();
  console.log({
    heapUsed: `${Math.round(used.heapUsed / 1024 / 1024)} MB`,
    heapTotal: `${Math.round(used.heapTotal / 1024 / 1024)} MB`,
    rss: `${Math.round(used.rss / 1024 / 1024)} MB`,
  });
}

// Heap snapshot
import v8 from 'v8';
import fs from 'fs';

function takeHeapSnapshot() {
  const snapshotPath = `heap-${Date.now()}.heapsnapshot`;
  const stream = fs.createWriteStream(snapshotPath);
  v8.writeHeapSnapshot(snapshotPath);
  console.log(`Heap snapshot written to ${snapshotPath}`);
}
```

## Tips

- Profile before optimizing
- Use streams for large data
- Implement caching at multiple levels
- Use connection pooling for databases
- Leverage clustering for CPU-bound work
- Monitor memory usage in production
- Use worker threads for heavy computation
- Batch database operations
- Implement proper error handling
- Use async/await consistently
