---
name: security-best-practices
description: Web application security best practices including authentication, authorization, input validation, XSS/CSRF prevention, and secure coding patterns.
---

# Security Best Practices

Web application security best practices including authentication, authorization, input validation, XSS/CSRF prevention, and secure coding patterns.

## Authentication

### Password Hashing

```typescript
import bcrypt from 'bcrypt';

const SALT_ROUNDS = 12;

// Hash password
async function hashPassword(password: string): Promise<string> {
  return bcrypt.hash(password, SALT_ROUNDS);
}

// Verify password
async function verifyPassword(password: string, hash: string): Promise<boolean> {
  return bcrypt.compare(password, hash);
}

// Usage
const hash = await hashPassword('userPassword123');
const isValid = await verifyPassword('userPassword123', hash);
```

### JWT Implementation

```typescript
import jwt from 'jsonwebtoken';

interface TokenPayload {
  userId: string;
  email: string;
  role: string;
}

const JWT_SECRET = process.env.JWT_SECRET!;
const ACCESS_TOKEN_EXPIRY = '15m';
const REFRESH_TOKEN_EXPIRY = '7d';

function generateAccessToken(payload: TokenPayload): string {
  return jwt.sign(payload, JWT_SECRET, { expiresIn: ACCESS_TOKEN_EXPIRY });
}

function generateRefreshToken(userId: string): string {
  return jwt.sign({ userId }, JWT_SECRET, { expiresIn: REFRESH_TOKEN_EXPIRY });
}

function verifyToken(token: string): TokenPayload {
  try {
    return jwt.verify(token, JWT_SECRET) as TokenPayload;
  } catch (error) {
    throw new Error('Invalid token');
  }
}

// Middleware
function authMiddleware(req: Request, res: Response, next: NextFunction) {
  const authHeader = req.headers.authorization;

  if (!authHeader?.startsWith('Bearer ')) {
    return res.status(401).json({ error: 'Missing token' });
  }

  const token = authHeader.split(' ')[1];

  try {
    req.user = verifyToken(token);
    next();
  } catch {
    return res.status(401).json({ error: 'Invalid token' });
  }
}
```

### Secure Session Configuration

```typescript
import session from 'express-session';
import RedisStore from 'connect-redis';
import { createClient } from 'redis';

const redisClient = createClient({ url: process.env.REDIS_URL });

app.use(session({
  store: new RedisStore({ client: redisClient }),
  secret: process.env.SESSION_SECRET!,
  name: 'sessionId', // Don't use default 'connect.sid'
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: process.env.NODE_ENV === 'production', // HTTPS only
    httpOnly: true, // Prevent XSS access
    sameSite: 'strict', // CSRF protection
    maxAge: 24 * 60 * 60 * 1000, // 24 hours
  },
}));
```

## Authorization

### Role-Based Access Control (RBAC)

```typescript
type Role = 'admin' | 'moderator' | 'user';
type Permission = 'read' | 'write' | 'delete' | 'manage_users';

const rolePermissions: Record<Role, Permission[]> = {
  admin: ['read', 'write', 'delete', 'manage_users'],
  moderator: ['read', 'write', 'delete'],
  user: ['read', 'write'],
};

function hasPermission(userRole: Role, permission: Permission): boolean {
  return rolePermissions[userRole]?.includes(permission) ?? false;
}

// Middleware
function requirePermission(permission: Permission) {
  return (req: Request, res: Response, next: NextFunction) => {
    if (!req.user) {
      return res.status(401).json({ error: 'Unauthorized' });
    }

    if (!hasPermission(req.user.role, permission)) {
      return res.status(403).json({ error: 'Forbidden' });
    }

    next();
  };
}

// Usage
app.delete('/api/posts/:id', requirePermission('delete'), deletePost);
app.get('/api/admin/users', requirePermission('manage_users'), listUsers);
```

### Resource-Based Authorization

```typescript
async function canAccessResource(
  userId: string,
  resourceId: string,
  action: 'read' | 'write' | 'delete'
): Promise<boolean> {
  const resource = await db.posts.findById(resourceId);

  if (!resource) return false;

  // Owner can do anything
  if (resource.authorId === userId) return true;

  // Check sharing permissions
  const sharing = await db.sharing.findOne({
    resourceId,
    userId,
  });

  if (!sharing) return false;

  switch (action) {
    case 'read':
      return ['read', 'write'].includes(sharing.permission);
    case 'write':
      return sharing.permission === 'write';
    case 'delete':
      return false; // Only owner can delete
    default:
      return false;
  }
}

// Middleware
function authorizeResource(action: 'read' | 'write' | 'delete') {
  return async (req: Request, res: Response, next: NextFunction) => {
    const canAccess = await canAccessResource(
      req.user.id,
      req.params.id,
      action
    );

    if (!canAccess) {
      return res.status(403).json({ error: 'Access denied' });
    }

    next();
  };
}
```

## Input Validation

### Schema Validation with Zod

```typescript
import { z } from 'zod';

const userSchema = z.object({
  email: z.string().email('Invalid email format'),
  password: z
    .string()
    .min(8, 'Password must be at least 8 characters')
    .regex(/[A-Z]/, 'Password must contain uppercase letter')
    .regex(/[0-9]/, 'Password must contain number'),
  name: z.string().min(1).max(100),
  age: z.number().int().positive().optional(),
});

// Validation middleware
function validate<T>(schema: z.Schema<T>) {
  return (req: Request, res: Response, next: NextFunction) => {
    try {
      req.body = schema.parse(req.body);
      next();
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({
          error: 'Validation failed',
          details: error.errors,
        });
      }
      next(error);
    }
  };
}

// Usage
app.post('/api/users', validate(userSchema), createUser);
```

### SQL Injection Prevention

```typescript
// BAD - Vulnerable to SQL injection
const query = `SELECT * FROM users WHERE email = '${email}'`;

// GOOD - Parameterized query
const query = 'SELECT * FROM users WHERE email = $1';
const result = await db.query(query, [email]);

// GOOD - Using ORM (Prisma)
const user = await prisma.user.findUnique({
  where: { email },
});

// GOOD - Using query builder (Knex)
const user = await knex('users').where({ email }).first();
```

## XSS Prevention

### Output Encoding

```typescript
import DOMPurify from 'dompurify';
import { JSDOM } from 'jsdom';

const window = new JSDOM('').window;
const purify = DOMPurify(window);

// Sanitize HTML input
function sanitizeHTML(dirty: string): string {
  return purify.sanitize(dirty, {
    ALLOWED_TAGS: ['b', 'i', 'em', 'strong', 'a', 'p', 'br'],
    ALLOWED_ATTR: ['href'],
  });
}

// Escape for HTML context
function escapeHTML(str: string): string {
  const escapeMap: Record<string, string> = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    "'": '&#x27;',
  };
  return str.replace(/[&<>"']/g, char => escapeMap[char]);
}
```

### Content Security Policy

```typescript
import helmet from 'helmet';

app.use(helmet());

app.use(helmet.contentSecurityPolicy({
  directives: {
    defaultSrc: ["'self'"],
    scriptSrc: ["'self'", "'strict-dynamic'"],
    styleSrc: ["'self'", "'unsafe-inline'"],
    imgSrc: ["'self'", 'data:', 'https:'],
    fontSrc: ["'self'"],
    connectSrc: ["'self'", 'https://api.example.com'],
    frameSrc: ["'none'"],
    objectSrc: ["'none'"],
    upgradeInsecureRequests: [],
  },
}));
```

## CSRF Protection

```typescript
import csrf from 'csurf';

// Enable CSRF protection
const csrfProtection = csrf({
  cookie: {
    httpOnly: true,
    secure: process.env.NODE_ENV === 'production',
    sameSite: 'strict',
  },
});

app.use(csrfProtection);

// Provide token to frontend
app.get('/api/csrf-token', (req, res) => {
  res.json({ csrfToken: req.csrfToken() });
});

// Frontend: Include token in requests
fetch('/api/data', {
  method: 'POST',
  headers: {
    'Content-Type': 'application/json',
    'CSRF-Token': csrfToken,
  },
  body: JSON.stringify(data),
});
```

## Rate Limiting

```typescript
import rateLimit from 'express-rate-limit';
import RedisStore from 'rate-limit-redis';

// General rate limit
const generalLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // 100 requests per window
  standardHeaders: true,
  legacyHeaders: false,
  message: { error: 'Too many requests' },
});

// Strict limit for auth endpoints
const authLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour
  max: 5, // 5 attempts
  message: { error: 'Too many login attempts' },
  store: new RedisStore({
    client: redisClient,
    prefix: 'rl:auth:',
  }),
});

app.use('/api', generalLimiter);
app.use('/api/auth/login', authLimiter);
app.use('/api/auth/register', authLimiter);
```

## Security Headers

```typescript
import helmet from 'helmet';

app.use(helmet());

// Custom headers
app.use((req, res, next) => {
  // Prevent clickjacking
  res.setHeader('X-Frame-Options', 'DENY');

  // XSS protection
  res.setHeader('X-XSS-Protection', '1; mode=block');

  // Prevent MIME sniffing
  res.setHeader('X-Content-Type-Options', 'nosniff');

  // Referrer policy
  res.setHeader('Referrer-Policy', 'strict-origin-when-cross-origin');

  // Permissions policy
  res.setHeader('Permissions-Policy', 'geolocation=(), microphone=()');

  next();
});
```

## Secrets Management

```typescript
// DON'T: Hardcode secrets
const API_KEY = 'sk_live_abc123';

// DO: Use environment variables
const API_KEY = process.env.API_KEY;

// DO: Validate required env vars at startup
const requiredEnvVars = ['DATABASE_URL', 'JWT_SECRET', 'API_KEY'];

for (const envVar of requiredEnvVars) {
  if (!process.env[envVar]) {
    throw new Error(`Missing required environment variable: ${envVar}`);
  }
}

// DO: Use secrets manager in production
import { SecretManagerServiceClient } from '@google-cloud/secret-manager';

const client = new SecretManagerServiceClient();

async function getSecret(name: string): Promise<string> {
  const [version] = await client.accessSecretVersion({
    name: `projects/my-project/secrets/${name}/versions/latest`,
  });
  return version.payload?.data?.toString() ?? '';
}
```

## Security Checklist

- [ ] Use HTTPS everywhere
- [ ] Hash passwords with bcrypt (cost factor 12+)
- [ ] Implement proper session management
- [ ] Validate and sanitize all inputs
- [ ] Use parameterized queries
- [ ] Implement CSRF protection
- [ ] Set security headers (use Helmet.js)
- [ ] Implement rate limiting
- [ ] Keep dependencies updated
- [ ] Log security events
- [ ] Use CSP to prevent XSS
- [ ] Implement proper error handling (don't leak info)
- [ ] Regular security audits

## Tips

- Never trust user input
- Implement defense in depth
- Follow the principle of least privilege
- Keep security libraries updated
- Use established libraries, don't roll your own crypto
- Log security events for monitoring
- Have an incident response plan
