---
name: testing-strategies
description: Comprehensive testing strategies including unit tests, integration tests, E2E tests, mocking, test organization, and coverage best practices.
---

# Testing Strategies

Comprehensive testing strategies including unit tests, integration tests, E2E tests, mocking, test organization, and coverage best practices.

## Testing Pyramid

```
        /\
       /  \     E2E Tests (Few)
      /----\    - Full user flows
     /      \   - Slow, expensive
    /--------\  Integration Tests (Some)
   /          \ - Component interactions
  /            \- API + DB tests
 /--------------\  Unit Tests (Many)
/                \ - Fast, isolated
/------------------\- Pure functions
```

## Unit Testing

### Jest Basics

```typescript
// math.ts
export function add(a: number, b: number): number {
  return a + b;
}

export function divide(a: number, b: number): number {
  if (b === 0) throw new Error('Division by zero');
  return a / b;
}

// math.test.ts
import { add, divide } from './math';

describe('math utilities', () => {
  describe('add', () => {
    it('adds two positive numbers', () => {
      expect(add(2, 3)).toBe(5);
    });

    it('handles negative numbers', () => {
      expect(add(-1, 1)).toBe(0);
    });

    it('handles decimals', () => {
      expect(add(0.1, 0.2)).toBeCloseTo(0.3);
    });
  });

  describe('divide', () => {
    it('divides two numbers', () => {
      expect(divide(10, 2)).toBe(5);
    });

    it('throws on division by zero', () => {
      expect(() => divide(10, 0)).toThrow('Division by zero');
    });
  });
});
```

### Testing Async Code

```typescript
// userService.ts
export async function fetchUser(id: string): Promise<User> {
  const response = await fetch(`/api/users/${id}`);
  if (!response.ok) throw new Error('User not found');
  return response.json();
}

// userService.test.ts
import { fetchUser } from './userService';

// Mock fetch globally
global.fetch = jest.fn();

describe('fetchUser', () => {
  beforeEach(() => {
    jest.clearAllMocks();
  });

  it('returns user data on success', async () => {
    const mockUser = { id: '1', name: 'John' };

    (fetch as jest.Mock).mockResolvedValueOnce({
      ok: true,
      json: async () => mockUser,
    });

    const user = await fetchUser('1');
    expect(user).toEqual(mockUser);
    expect(fetch).toHaveBeenCalledWith('/api/users/1');
  });

  it('throws on error response', async () => {
    (fetch as jest.Mock).mockResolvedValueOnce({ ok: false });

    await expect(fetchUser('999')).rejects.toThrow('User not found');
  });
});
```

### Mocking Modules

```typescript
// database.ts
import { db } from './db';

export async function getUserCount(): Promise<number> {
  const result = await db.query('SELECT COUNT(*) FROM users');
  return result.rows[0].count;
}

// database.test.ts
jest.mock('./db', () => ({
  db: {
    query: jest.fn(),
  },
}));

import { getUserCount } from './database';
import { db } from './db';

describe('getUserCount', () => {
  it('returns the user count', async () => {
    (db.query as jest.Mock).mockResolvedValueOnce({
      rows: [{ count: 42 }],
    });

    const count = await getUserCount();
    expect(count).toBe(42);
  });
});
```

## React Testing

### React Testing Library

```tsx
// Button.tsx
interface ButtonProps {
  onClick: () => void;
  disabled?: boolean;
  children: React.ReactNode;
}

export function Button({ onClick, disabled, children }: ButtonProps) {
  return (
    <button onClick={onClick} disabled={disabled}>
      {children}
    </button>
  );
}

// Button.test.tsx
import { render, screen, fireEvent } from '@testing-library/react';
import userEvent from '@testing-library/user-event';
import { Button } from './Button';

describe('Button', () => {
  it('renders children', () => {
    render(<Button onClick={() => {}}>Click me</Button>);
    expect(screen.getByRole('button', { name: /click me/i })).toBeInTheDocument();
  });

  it('calls onClick when clicked', async () => {
    const handleClick = jest.fn();
    render(<Button onClick={handleClick}>Click</Button>);

    await userEvent.click(screen.getByRole('button'));
    expect(handleClick).toHaveBeenCalledTimes(1);
  });

  it('does not call onClick when disabled', async () => {
    const handleClick = jest.fn();
    render(<Button onClick={handleClick} disabled>Click</Button>);

    await userEvent.click(screen.getByRole('button'));
    expect(handleClick).not.toHaveBeenCalled();
  });
});
```

### Testing Hooks

```tsx
// useCounter.ts
import { useState, useCallback } from 'react';

export function useCounter(initial = 0) {
  const [count, setCount] = useState(initial);

  const increment = useCallback(() => setCount(c => c + 1), []);
  const decrement = useCallback(() => setCount(c => c - 1), []);
  const reset = useCallback(() => setCount(initial), [initial]);

  return { count, increment, decrement, reset };
}

// useCounter.test.ts
import { renderHook, act } from '@testing-library/react';
import { useCounter } from './useCounter';

describe('useCounter', () => {
  it('initializes with default value', () => {
    const { result } = renderHook(() => useCounter());
    expect(result.current.count).toBe(0);
  });

  it('initializes with provided value', () => {
    const { result } = renderHook(() => useCounter(10));
    expect(result.current.count).toBe(10);
  });

  it('increments count', () => {
    const { result } = renderHook(() => useCounter());

    act(() => {
      result.current.increment();
    });

    expect(result.current.count).toBe(1);
  });

  it('resets to initial value', () => {
    const { result } = renderHook(() => useCounter(5));

    act(() => {
      result.current.increment();
      result.current.increment();
      result.current.reset();
    });

    expect(result.current.count).toBe(5);
  });
});
```

## Integration Testing

### API Integration Tests

```typescript
// app.test.ts
import request from 'supertest';
import { app } from './app';
import { db } from './db';

describe('Users API', () => {
  beforeAll(async () => {
    await db.migrate.latest();
  });

  beforeEach(async () => {
    await db('users').truncate();
  });

  afterAll(async () => {
    await db.destroy();
  });

  describe('GET /api/users', () => {
    it('returns empty array when no users', async () => {
      const response = await request(app)
        .get('/api/users')
        .expect(200);

      expect(response.body).toEqual([]);
    });

    it('returns all users', async () => {
      await db('users').insert([
        { name: 'John', email: 'john@test.com' },
        { name: 'Jane', email: 'jane@test.com' },
      ]);

      const response = await request(app)
        .get('/api/users')
        .expect(200);

      expect(response.body).toHaveLength(2);
    });
  });

  describe('POST /api/users', () => {
    it('creates a new user', async () => {
      const newUser = { name: 'John', email: 'john@test.com' };

      const response = await request(app)
        .post('/api/users')
        .send(newUser)
        .expect(201);

      expect(response.body).toMatchObject(newUser);
      expect(response.body.id).toBeDefined();
    });

    it('returns 400 for invalid data', async () => {
      const response = await request(app)
        .post('/api/users')
        .send({ name: 'John' }) // Missing email
        .expect(400);

      expect(response.body.error).toBeDefined();
    });
  });
});
```

## E2E Testing with Playwright

```typescript
// e2e/auth.spec.ts
import { test, expect } from '@playwright/test';

test.describe('Authentication', () => {
  test('user can sign up', async ({ page }) => {
    await page.goto('/signup');

    await page.fill('[name="email"]', 'test@example.com');
    await page.fill('[name="password"]', 'password123');
    await page.fill('[name="confirmPassword"]', 'password123');

    await page.click('button[type="submit"]');

    await expect(page).toHaveURL('/dashboard');
    await expect(page.locator('text=Welcome')).toBeVisible();
  });

  test('user can log in', async ({ page }) => {
    // Seed test user
    await page.goto('/login');

    await page.fill('[name="email"]', 'existing@example.com');
    await page.fill('[name="password"]', 'password123');

    await page.click('button[type="submit"]');

    await expect(page).toHaveURL('/dashboard');
  });

  test('shows error for invalid credentials', async ({ page }) => {
    await page.goto('/login');

    await page.fill('[name="email"]', 'wrong@example.com');
    await page.fill('[name="password"]', 'wrongpassword');

    await page.click('button[type="submit"]');

    await expect(page.locator('text=Invalid credentials')).toBeVisible();
    await expect(page).toHaveURL('/login');
  });
});
```

## Test Organization

### File Structure

```
src/
├── components/
│   ├── Button/
│   │   ├── Button.tsx
│   │   ├── Button.test.tsx
│   │   └── Button.stories.tsx
│   └── Form/
│       ├── Form.tsx
│       └── Form.test.tsx
├── hooks/
│   ├── useAuth.ts
│   └── useAuth.test.ts
├── utils/
│   ├── validation.ts
│   └── validation.test.ts
└── __tests__/
    └── integration/
        └── api.test.ts

e2e/
├── auth.spec.ts
├── checkout.spec.ts
└── fixtures/
    └── users.json
```

### Test Patterns

```typescript
// Arrange-Act-Assert
test('calculates total correctly', () => {
  // Arrange
  const items = [
    { price: 10, quantity: 2 },
    { price: 5, quantity: 3 },
  ];

  // Act
  const total = calculateTotal(items);

  // Assert
  expect(total).toBe(35);
});

// Given-When-Then (BDD style)
describe('Shopping Cart', () => {
  describe('given items in cart', () => {
    describe('when applying discount code', () => {
      it('then reduces total by discount percentage', () => {
        const cart = new Cart([{ price: 100 }]);
        cart.applyDiscount('SAVE10');
        expect(cart.total).toBe(90);
      });
    });
  });
});
```

## Coverage Configuration

```javascript
// jest.config.js
module.exports = {
  collectCoverage: true,
  coverageDirectory: 'coverage',
  coverageReporters: ['text', 'lcov', 'html'],
  collectCoverageFrom: [
    'src/**/*.{ts,tsx}',
    '!src/**/*.d.ts',
    '!src/**/*.stories.tsx',
    '!src/index.tsx',
  ],
  coverageThreshold: {
    global: {
      branches: 80,
      functions: 80,
      lines: 80,
      statements: 80,
    },
  },
};
```

## Tips

- Write tests that describe behavior, not implementation
- Keep tests independent and isolated
- Use descriptive test names
- Follow the testing pyramid
- Mock external dependencies, not internal modules
- Use factories for test data creation
- Run tests in CI/CD pipeline
- Aim for meaningful coverage, not 100%
- Test edge cases and error conditions
- Keep tests fast and reliable
