---
name: typescript-patterns
description: Advanced TypeScript patterns including generics, utility types, type guards, discriminated unions, module augmentation, and type-safe API design.
---

# Advanced TypeScript Patterns

Advanced TypeScript patterns including generics, utility types, type guards, discriminated unions, module augmentation, and type-safe API design.

## Generics

### Basic Generics

```typescript
// Generic function
function identity<T>(value: T): T {
  return value;
}

// Generic interface
interface Container<T> {
  value: T;
  getValue(): T;
}

// Generic class
class Box<T> {
  constructor(private content: T) {}

  getContent(): T {
    return this.content;
  }
}

// Multiple type parameters
function pair<T, U>(first: T, second: U): [T, U] {
  return [first, second];
}
```

### Generic Constraints

```typescript
// Constrain to types with specific properties
interface HasLength {
  length: number;
}

function logLength<T extends HasLength>(item: T): void {
  console.log(item.length);
}

// Constrain to object keys
function getProperty<T, K extends keyof T>(obj: T, key: K): T[K] {
  return obj[key];
}

// Default type parameter
interface PaginatedResponse<T = unknown> {
  data: T[];
  page: number;
  total: number;
}
```

## Utility Types

### Built-in Utility Types

```typescript
interface User {
  id: string;
  name: string;
  email: string;
  age: number;
  role: 'admin' | 'user';
}

// Partial - all properties optional
type PartialUser = Partial<User>;

// Required - all properties required
type RequiredUser = Required<PartialUser>;

// Readonly - all properties readonly
type ReadonlyUser = Readonly<User>;

// Pick - select specific properties
type UserCredentials = Pick<User, 'email' | 'id'>;

// Omit - exclude specific properties
type PublicUser = Omit<User, 'email' | 'role'>;

// Record - construct object type
type UserRoles = Record<string, User>;

// Extract - extract union members
type AdminRole = Extract<User['role'], 'admin'>;

// Exclude - exclude union members
type NonAdminRole = Exclude<User['role'], 'admin'>;

// NonNullable - remove null and undefined
type NonNullString = NonNullable<string | null | undefined>;

// ReturnType - extract function return type
function createUser() { return { id: '1', name: 'John' }; }
type CreatedUser = ReturnType<typeof createUser>;

// Parameters - extract function parameters
type CreateUserParams = Parameters<typeof createUser>;
```

### Custom Utility Types

```typescript
// Deep Partial
type DeepPartial<T> = {
  [P in keyof T]?: T[P] extends object ? DeepPartial<T[P]> : T[P];
};

// Deep Readonly
type DeepReadonly<T> = {
  readonly [P in keyof T]: T[P] extends object ? DeepReadonly<T[P]> : T[P];
};

// Nullable
type Nullable<T> = T | null;

// Optional keys
type OptionalKeys<T, K extends keyof T> = Omit<T, K> & Partial<Pick<T, K>>;

// Required keys
type RequiredKeys<T, K extends keyof T> = Omit<T, K> & Required<Pick<T, K>>;

// Mutable (remove readonly)
type Mutable<T> = {
  -readonly [P in keyof T]: T[P];
};
```

## Type Guards

### typeof Guards

```typescript
function processValue(value: string | number) {
  if (typeof value === 'string') {
    return value.toUpperCase(); // string methods available
  }
  return value.toFixed(2); // number methods available
}
```

### instanceof Guards

```typescript
class ApiError extends Error {
  constructor(public code: number, message: string) {
    super(message);
  }
}

function handleError(error: Error) {
  if (error instanceof ApiError) {
    console.log(`API Error ${error.code}: ${error.message}`);
  } else {
    console.log(`Error: ${error.message}`);
  }
}
```

### Custom Type Guards

```typescript
interface Cat {
  type: 'cat';
  meow(): void;
}

interface Dog {
  type: 'dog';
  bark(): void;
}

type Pet = Cat | Dog;

// Type predicate
function isCat(pet: Pet): pet is Cat {
  return pet.type === 'cat';
}

function handlePet(pet: Pet) {
  if (isCat(pet)) {
    pet.meow(); // TypeScript knows it's a Cat
  } else {
    pet.bark(); // TypeScript knows it's a Dog
  }
}

// Assertion function
function assertIsString(value: unknown): asserts value is string {
  if (typeof value !== 'string') {
    throw new Error('Value must be a string');
  }
}
```

## Discriminated Unions

```typescript
// Define discriminated union
interface LoadingState {
  status: 'loading';
}

interface SuccessState<T> {
  status: 'success';
  data: T;
}

interface ErrorState {
  status: 'error';
  error: string;
}

type AsyncState<T> = LoadingState | SuccessState<T> | ErrorState;

// Exhaustive handling
function handleState<T>(state: AsyncState<T>): string {
  switch (state.status) {
    case 'loading':
      return 'Loading...';
    case 'success':
      return `Data: ${JSON.stringify(state.data)}`;
    case 'error':
      return `Error: ${state.error}`;
    default:
      // Exhaustiveness check
      const _exhaustive: never = state;
      return _exhaustive;
  }
}
```

## Mapped Types

```typescript
// Basic mapped type
type Getters<T> = {
  [K in keyof T as `get${Capitalize<string & K>}`]: () => T[K];
};

interface Person {
  name: string;
  age: number;
}

type PersonGetters = Getters<Person>;
// { getName: () => string; getAge: () => number; }

// Conditional mapped type
type ReadonlyIfObject<T> = {
  [K in keyof T]: T[K] extends object ? Readonly<T[K]> : T[K];
};

// Filter keys by value type
type FilterByType<T, U> = {
  [K in keyof T as T[K] extends U ? K : never]: T[K];
};

type StringProps = FilterByType<Person, string>;
// { name: string; }
```

## Template Literal Types

```typescript
// Basic template literals
type EventName = `on${Capitalize<'click' | 'focus' | 'blur'>}`;
// "onClick" | "onFocus" | "onBlur"

// Dynamic property names
type PropGetters<T> = {
  [K in keyof T as `get${Capitalize<string & K>}`]: () => T[K];
} & {
  [K in keyof T as `set${Capitalize<string & K>}`]: (value: T[K]) => void;
};

// CSS unit types
type CSSUnit = 'px' | 'em' | 'rem' | '%';
type CSSValue = `${number}${CSSUnit}`;

const width: CSSValue = '100px';
const padding: CSSValue = '1.5rem';
```

## Module Augmentation

```typescript
// Extend existing module types
declare module 'express' {
  interface Request {
    user?: {
      id: string;
      email: string;
    };
  }
}

// Extend global types
declare global {
  interface Window {
    analytics: {
      track(event: string, data?: Record<string, unknown>): void;
    };
  }
}

// Extend third-party library
declare module 'axios' {
  export interface AxiosRequestConfig {
    retry?: boolean;
    retryCount?: number;
  }
}
```

## Type-Safe API Design

### Type-Safe Fetch Wrapper

```typescript
interface ApiResponse<T> {
  data: T;
  status: number;
}

interface ApiError {
  message: string;
  code: string;
}

type Result<T> =
  | { success: true; data: T }
  | { success: false; error: ApiError };

async function fetchApi<T>(
  url: string,
  options?: RequestInit
): Promise<Result<T>> {
  try {
    const response = await fetch(url, options);
    const data = await response.json();

    if (!response.ok) {
      return { success: false, error: data as ApiError };
    }

    return { success: true, data: data as T };
  } catch (error) {
    return {
      success: false,
      error: { message: 'Network error', code: 'NETWORK_ERROR' }
    };
  }
}

// Usage
interface User {
  id: string;
  name: string;
}

const result = await fetchApi<User>('/api/users/1');

if (result.success) {
  console.log(result.data.name); // Type-safe access
} else {
  console.error(result.error.message);
}
```

### Type-Safe Event Emitter

```typescript
type EventMap = {
  userCreated: { id: string; name: string };
  userDeleted: { id: string };
  error: { message: string };
};

class TypedEventEmitter<T extends Record<string, unknown>> {
  private handlers = new Map<keyof T, Set<(data: unknown) => void>>();

  on<K extends keyof T>(event: K, handler: (data: T[K]) => void): void {
    if (!this.handlers.has(event)) {
      this.handlers.set(event, new Set());
    }
    this.handlers.get(event)!.add(handler as (data: unknown) => void);
  }

  emit<K extends keyof T>(event: K, data: T[K]): void {
    this.handlers.get(event)?.forEach(handler => handler(data));
  }
}

const emitter = new TypedEventEmitter<EventMap>();

emitter.on('userCreated', (data) => {
  console.log(data.name); // Type-safe: { id: string; name: string }
});

emitter.emit('userCreated', { id: '1', name: 'John' });
```

## Tips

- Use `unknown` instead of `any` for type safety
- Prefer interfaces for object shapes, types for unions
- Use discriminated unions for state management
- Leverage mapped types for DRY type definitions
- Use assertion functions for runtime validation
- Export types alongside functions for consumers
- Use `satisfies` operator for type checking without widening
