---
name: api-design
description: RESTful API design patterns with OpenAPI specification, authentication strategies, versioning, error handling, pagination, and rate limiting best practices.
---

# RESTful API Design

RESTful API design patterns with OpenAPI specification, authentication strategies, versioning, error handling, pagination, and rate limiting best practices.

## REST Principles

### Resource Naming

```
# Good - Nouns, plural, hierarchical
GET    /users
GET    /users/123
GET    /users/123/orders
GET    /users/123/orders/456
POST   /users
PUT    /users/123
DELETE /users/123

# Bad - Verbs, actions in URL
GET    /getUsers
POST   /createUser
GET    /getUserOrders
```

### HTTP Methods

| Method | Purpose | Idempotent | Safe |
|--------|---------|------------|------|
| GET | Retrieve resource | Yes | Yes |
| POST | Create resource | No | No |
| PUT | Replace resource | Yes | No |
| PATCH | Partial update | No | No |
| DELETE | Remove resource | Yes | No |

### Status Codes

```
# Success
200 OK              - Successful GET/PUT/PATCH
201 Created         - Successful POST (include Location header)
204 No Content      - Successful DELETE

# Client Errors
400 Bad Request     - Invalid input
401 Unauthorized    - Missing/invalid authentication
403 Forbidden       - Authenticated but not authorized
404 Not Found       - Resource doesn't exist
409 Conflict        - State conflict (duplicate, etc.)
422 Unprocessable   - Validation errors
429 Too Many Reqs   - Rate limit exceeded

# Server Errors
500 Internal Error  - Unexpected server error
502 Bad Gateway     - Upstream service error
503 Unavailable     - Service temporarily down
```

## OpenAPI Specification

```yaml
openapi: 3.0.3
info:
  title: User API
  version: 1.0.0
  description: User management API

servers:
  - url: https://api.example.com/v1
    description: Production
  - url: https://staging-api.example.com/v1
    description: Staging

paths:
  /users:
    get:
      summary: List users
      tags: [Users]
      parameters:
        - name: page
          in: query
          schema:
            type: integer
            default: 1
        - name: limit
          in: query
          schema:
            type: integer
            default: 20
            maximum: 100
      responses:
        '200':
          description: List of users
          content:
            application/json:
              schema:
                $ref: '#/components/schemas/UserList'

    post:
      summary: Create user
      tags: [Users]
      requestBody:
        required: true
        content:
          application/json:
            schema:
              $ref: '#/components/schemas/CreateUser'
      responses:
        '201':
          description: User created
          headers:
            Location:
              schema:
                type: string
          content:
            application/json:
              schema:
                $ref: '#/components/schemas/User'
        '422':
          $ref: '#/components/responses/ValidationError'

  /users/{id}:
    get:
      summary: Get user by ID
      tags: [Users]
      parameters:
        - name: id
          in: path
          required: true
          schema:
            type: string
            format: uuid
      responses:
        '200':
          description: User details
          content:
            application/json:
              schema:
                $ref: '#/components/schemas/User'
        '404':
          $ref: '#/components/responses/NotFound'

components:
  schemas:
    User:
      type: object
      properties:
        id:
          type: string
          format: uuid
        email:
          type: string
          format: email
        name:
          type: string
        created_at:
          type: string
          format: date-time
      required: [id, email, name]

    CreateUser:
      type: object
      properties:
        email:
          type: string
          format: email
        name:
          type: string
        password:
          type: string
          minLength: 8
      required: [email, name, password]

    UserList:
      type: object
      properties:
        data:
          type: array
          items:
            $ref: '#/components/schemas/User'
        pagination:
          $ref: '#/components/schemas/Pagination'

    Pagination:
      type: object
      properties:
        page:
          type: integer
        limit:
          type: integer
        total:
          type: integer
        total_pages:
          type: integer

    Error:
      type: object
      properties:
        code:
          type: string
        message:
          type: string
        details:
          type: object

  responses:
    NotFound:
      description: Resource not found
      content:
        application/json:
          schema:
            $ref: '#/components/schemas/Error'
          example:
            code: NOT_FOUND
            message: User not found

    ValidationError:
      description: Validation error
      content:
        application/json:
          schema:
            $ref: '#/components/schemas/Error'

  securitySchemes:
    bearerAuth:
      type: http
      scheme: bearer
      bearerFormat: JWT

security:
  - bearerAuth: []
```

## Authentication

### JWT Bearer Token

```typescript
// Request
GET /users HTTP/1.1
Authorization: Bearer eyJhbGciOiJIUzI1NiIs...

// JWT Payload
{
  "sub": "user-123",
  "email": "user@example.com",
  "role": "admin",
  "iat": 1699900000,
  "exp": 1699903600
}
```

### API Keys

```typescript
// Header-based (preferred)
GET /users HTTP/1.1
X-API-Key: sk_live_abc123...

// Query parameter (less secure)
GET /users?api_key=sk_live_abc123
```

## Error Handling

### Consistent Error Format

```json
{
  "error": {
    "code": "VALIDATION_ERROR",
    "message": "Request validation failed",
    "details": {
      "fields": [
        {
          "field": "email",
          "message": "Invalid email format"
        },
        {
          "field": "password",
          "message": "Must be at least 8 characters"
        }
      ]
    },
    "request_id": "req_abc123",
    "documentation_url": "https://docs.api.com/errors/VALIDATION_ERROR"
  }
}
```

### Error Codes

```typescript
const ERROR_CODES = {
  // Authentication
  INVALID_TOKEN: 'Authentication token is invalid',
  TOKEN_EXPIRED: 'Authentication token has expired',

  // Authorization
  FORBIDDEN: 'You do not have permission to access this resource',

  // Validation
  VALIDATION_ERROR: 'Request validation failed',
  INVALID_FORMAT: 'Request body format is invalid',

  // Resources
  NOT_FOUND: 'Resource not found',
  ALREADY_EXISTS: 'Resource already exists',
  CONFLICT: 'Resource state conflict',

  // Rate Limiting
  RATE_LIMITED: 'Too many requests',

  // Server
  INTERNAL_ERROR: 'An unexpected error occurred',
  SERVICE_UNAVAILABLE: 'Service temporarily unavailable'
};
```

## Pagination

### Offset-based

```json
GET /users?page=2&limit=20

{
  "data": [...],
  "pagination": {
    "page": 2,
    "limit": 20,
    "total": 156,
    "total_pages": 8,
    "has_next": true,
    "has_prev": true
  }
}
```

### Cursor-based (Better for large datasets)

```json
GET /users?cursor=eyJpZCI6MTIzfQ&limit=20

{
  "data": [...],
  "pagination": {
    "next_cursor": "eyJpZCI6MTQzfQ",
    "prev_cursor": "eyJpZCI6MTAzfQ",
    "has_next": true,
    "has_prev": true
  }
}
```

## Versioning

### URL Path (Recommended)

```
https://api.example.com/v1/users
https://api.example.com/v2/users
```

### Header-based

```
GET /users HTTP/1.1
Accept: application/vnd.api+json; version=2
```

## Rate Limiting

### Response Headers

```
HTTP/1.1 200 OK
X-RateLimit-Limit: 1000
X-RateLimit-Remaining: 998
X-RateLimit-Reset: 1699900000
```

### Rate Limit Exceeded

```
HTTP/1.1 429 Too Many Requests
Retry-After: 60

{
  "error": {
    "code": "RATE_LIMITED",
    "message": "Rate limit exceeded. Try again in 60 seconds."
  }
}
```

## Filtering & Sorting

```
# Filtering
GET /users?status=active&role=admin
GET /users?created_at[gte]=2024-01-01
GET /users?email[contains]=@company.com

# Sorting
GET /users?sort=created_at:desc
GET /users?sort=name:asc,created_at:desc

# Field Selection
GET /users?fields=id,name,email

# Combining
GET /users?status=active&sort=created_at:desc&fields=id,name&limit=10
```

## Tips

- Use nouns for resources, HTTP methods for actions
- Return appropriate status codes
- Provide consistent error responses
- Document with OpenAPI/Swagger
- Version your API from the start
- Implement pagination for list endpoints
- Use cursor pagination for large datasets
- Include rate limit headers
- Support filtering and sorting
- Use HATEOAS for discoverability (optional)
