---
name: docker-compose
description: Container orchestration with Docker Compose for multi-service applications, including networking, volumes, environment configuration, and production deployment patterns.
---

# Docker Compose Orchestration

Container orchestration with Docker Compose for multi-service applications, including networking, volumes, environment configuration, and production deployment patterns.

## Basic Structure

```yaml
# docker-compose.yml
version: '3.8'

services:
  app:
    build: .
    ports:
      - "3000:3000"
    environment:
      - NODE_ENV=production
    depends_on:
      - db
      - redis

  db:
    image: postgres:15
    volumes:
      - postgres_data:/var/lib/postgresql/data
    environment:
      POSTGRES_DB: myapp
      POSTGRES_USER: user
      POSTGRES_PASSWORD: ${DB_PASSWORD}

  redis:
    image: redis:7-alpine
    volumes:
      - redis_data:/data

volumes:
  postgres_data:
  redis_data:
```

## Service Configuration

### Build Options

```yaml
services:
  app:
    build:
      context: .
      dockerfile: Dockerfile.prod
      args:
        - NODE_VERSION=20
      target: production
      cache_from:
        - myapp:latest
    image: myapp:${VERSION:-latest}
```

### Port Mapping

```yaml
services:
  web:
    ports:
      - "80:80"           # HOST:CONTAINER
      - "443:443"
      - "127.0.0.1:8080:8080"  # Bind to localhost only
    expose:
      - "3000"            # Internal only (no host mapping)
```

### Environment Variables

```yaml
services:
  app:
    environment:
      - NODE_ENV=production
      - API_KEY=${API_KEY}      # From shell or .env
      - DEBUG=false
    env_file:
      - .env
      - .env.local
```

### Health Checks

```yaml
services:
  api:
    healthcheck:
      test: ["CMD", "curl", "-f", "http://localhost:3000/health"]
      interval: 30s
      timeout: 10s
      retries: 3
      start_period: 40s
```

### Resource Limits

```yaml
services:
  app:
    deploy:
      resources:
        limits:
          cpus: '0.50'
          memory: 512M
        reservations:
          cpus: '0.25'
          memory: 256M
```

## Networking

### Custom Networks

```yaml
services:
  frontend:
    networks:
      - frontend_net

  backend:
    networks:
      - frontend_net
      - backend_net

  db:
    networks:
      - backend_net

networks:
  frontend_net:
    driver: bridge
  backend_net:
    driver: bridge
    internal: true  # No external access
```

### Network Aliases

```yaml
services:
  api:
    networks:
      backend:
        aliases:
          - api-service
          - backend-api
```

## Volumes

### Named Volumes

```yaml
services:
  db:
    volumes:
      - postgres_data:/var/lib/postgresql/data

volumes:
  postgres_data:
    driver: local
```

### Bind Mounts

```yaml
services:
  app:
    volumes:
      - ./src:/app/src:ro          # Read-only
      - ./config:/app/config
      - /app/node_modules          # Anonymous volume (exclude)
```

### tmpfs Mounts

```yaml
services:
  app:
    tmpfs:
      - /tmp
      - /run
```

## Multi-Environment Setup

### Base + Override Pattern

```yaml
# docker-compose.yml (base)
services:
  app:
    build: .
    environment:
      - NODE_ENV=production

# docker-compose.override.yml (development - auto-loaded)
services:
  app:
    build:
      target: development
    volumes:
      - .:/app
    environment:
      - NODE_ENV=development
      - DEBUG=true
    ports:
      - "3000:3000"
      - "9229:9229"  # Debug port

# docker-compose.prod.yml
services:
  app:
    deploy:
      replicas: 3
    restart: always
```

**Usage:**
```bash
# Development (uses override automatically)
docker compose up

# Production
docker compose -f docker-compose.yml -f docker-compose.prod.yml up -d
```

## Common Patterns

### Web App with Database

```yaml
version: '3.8'

services:
  app:
    build: .
    ports:
      - "3000:3000"
    environment:
      DATABASE_URL: postgres://user:pass@db:5432/myapp
      REDIS_URL: redis://redis:6379
    depends_on:
      db:
        condition: service_healthy
      redis:
        condition: service_started

  db:
    image: postgres:15
    volumes:
      - postgres_data:/var/lib/postgresql/data
      - ./init.sql:/docker-entrypoint-initdb.d/init.sql
    environment:
      POSTGRES_DB: myapp
      POSTGRES_USER: user
      POSTGRES_PASSWORD: pass
    healthcheck:
      test: ["CMD-SHELL", "pg_isready -U user -d myapp"]
      interval: 5s
      timeout: 5s
      retries: 5

  redis:
    image: redis:7-alpine
    command: redis-server --appendonly yes
    volumes:
      - redis_data:/data

volumes:
  postgres_data:
  redis_data:
```

### Nginx Reverse Proxy

```yaml
version: '3.8'

services:
  nginx:
    image: nginx:alpine
    ports:
      - "80:80"
      - "443:443"
    volumes:
      - ./nginx.conf:/etc/nginx/nginx.conf:ro
      - ./certs:/etc/nginx/certs:ro
    depends_on:
      - app

  app:
    build: .
    expose:
      - "3000"
    deploy:
      replicas: 2
```

### Development with Hot Reload

```yaml
version: '3.8'

services:
  app:
    build:
      context: .
      target: development
    volumes:
      - .:/app
      - /app/node_modules
    ports:
      - "3000:3000"
      - "9229:9229"
    command: npm run dev
    environment:
      - NODE_ENV=development
      - CHOKIDAR_USEPOLLING=true  # For file watching in containers
```

## Useful Commands

```bash
# Start services
docker compose up -d

# View logs
docker compose logs -f app

# Execute command in container
docker compose exec app sh

# Scale service
docker compose up -d --scale app=3

# Rebuild and restart
docker compose up -d --build

# Stop and remove
docker compose down

# Remove with volumes
docker compose down -v

# View running services
docker compose ps

# Pull latest images
docker compose pull
```

## Tips

- Use `.env` for sensitive values, never commit secrets
- Always specify image versions, avoid `latest` in production
- Use health checks for dependent services
- Leverage build cache with `cache_from`
- Use `depends_on` with conditions for proper startup order
- Keep base compose file minimal, use overrides for environments
- Use named volumes for persistent data
- Set restart policies for production (`always` or `unless-stopped`)
