---
name: git-workflow
description: Professional Git workflow management with branching strategies, commit conventions, code review processes, and CI/CD integration best practices.
---

# Git Workflow Management

Professional Git workflow management with branching strategies, commit conventions, code review processes, and CI/CD integration.

## Branching Strategies

### Git Flow

```
main (production)
├── develop (integration)
│   ├── feature/user-auth
│   ├── feature/dashboard
│   └── feature/api-v2
├── release/1.2.0
└── hotfix/security-patch
```

**Branch Types:**
- `main` - Production-ready code
- `develop` - Integration branch for features
- `feature/*` - New features
- `release/*` - Release preparation
- `hotfix/*` - Production bug fixes

### GitHub Flow (Simpler)

```
main
├── feature/new-feature
├── fix/bug-fix
└── chore/update-deps
```

**Rules:**
1. `main` is always deployable
2. Create descriptive branch names
3. Open PR early for visibility
4. Merge only after review and CI passes

## Commit Conventions

### Conventional Commits

```
<type>(<scope>): <description>

[optional body]

[optional footer(s)]
```

**Types:**
| Type | Description |
|------|-------------|
| `feat` | New feature |
| `fix` | Bug fix |
| `docs` | Documentation |
| `style` | Formatting (no code change) |
| `refactor` | Code restructuring |
| `perf` | Performance improvement |
| `test` | Adding tests |
| `chore` | Maintenance tasks |
| `ci` | CI/CD changes |

**Examples:**
```bash
feat(auth): add OAuth2 login with Google
fix(api): handle null response from payment gateway
docs(readme): update installation instructions
refactor(utils): extract date formatting helpers
```

### Commit Message Best Practices

```bash
# Good
git commit -m "feat(cart): add quantity validation before checkout

- Validate quantity is positive integer
- Check stock availability
- Show user-friendly error messages

Closes #123"

# Bad
git commit -m "fixed stuff"
git commit -m "WIP"
```

## Common Workflows

### Starting a Feature

```bash
# Update develop branch
git checkout develop
git pull origin develop

# Create feature branch
git checkout -b feature/user-notifications

# Work on feature...
git add .
git commit -m "feat(notifications): add email notification service"

# Push and create PR
git push -u origin feature/user-notifications
gh pr create --base develop --title "Add user notifications"
```

### Code Review Process

```bash
# Fetch PR for local review
gh pr checkout 123

# Review changes
git diff develop...HEAD

# Add review comments
gh pr review 123 --comment --body "Looks good, minor suggestions inline"

# Approve
gh pr review 123 --approve

# Request changes
gh pr review 123 --request-changes --body "Please address security concerns"
```

### Handling Merge Conflicts

```bash
# Update your branch with latest changes
git checkout feature/my-feature
git fetch origin
git rebase origin/develop

# If conflicts occur:
# 1. Edit conflicted files
# 2. Mark as resolved
git add <conflicted-files>
git rebase --continue

# Force push (only on feature branches!)
git push --force-with-lease
```

### Creating a Release

```bash
# Create release branch
git checkout develop
git checkout -b release/1.2.0

# Update version
npm version 1.2.0 --no-git-tag-version
git commit -am "chore(release): bump version to 1.2.0"

# Merge to main
git checkout main
git merge --no-ff release/1.2.0
git tag -a v1.2.0 -m "Release 1.2.0"
git push origin main --tags

# Back-merge to develop
git checkout develop
git merge --no-ff release/1.2.0
git push origin develop

# Delete release branch
git branch -d release/1.2.0
```

### Hotfix Workflow

```bash
# Create hotfix from main
git checkout main
git checkout -b hotfix/security-patch

# Fix the issue
git commit -m "fix(auth): patch XSS vulnerability in login form"

# Merge to main
git checkout main
git merge --no-ff hotfix/security-patch
git tag -a v1.1.1 -m "Security patch"
git push origin main --tags

# Merge to develop
git checkout develop
git merge --no-ff hotfix/security-patch
git push origin develop
```

## Git Hooks

### Pre-commit Hook

```bash
#!/bin/sh
# .git/hooks/pre-commit

# Run linter
npm run lint
if [ $? -ne 0 ]; then
  echo "Linting failed. Please fix errors before committing."
  exit 1
fi

# Run tests
npm run test:unit
if [ $? -ne 0 ]; then
  echo "Tests failed. Please fix before committing."
  exit 1
fi
```

### Commit-msg Hook

```bash
#!/bin/sh
# .git/hooks/commit-msg

commit_regex='^(feat|fix|docs|style|refactor|perf|test|chore|ci)(\(.+\))?: .{1,50}'

if ! grep -qE "$commit_regex" "$1"; then
  echo "Invalid commit message format."
  echo "Use: <type>(<scope>): <description>"
  exit 1
fi
```

## Useful Git Aliases

```bash
# Add to ~/.gitconfig
[alias]
  co = checkout
  br = branch
  ci = commit
  st = status
  lg = log --oneline --graph --decorate
  last = log -1 HEAD
  unstage = reset HEAD --
  amend = commit --amend --no-edit
  wip = commit -am "WIP"
  undo = reset --soft HEAD~1
  branches = branch -a
  tags = tag -l
  stashes = stash list
  discard = checkout --
  contributors = shortlog -sn
```

## Tips

- Keep commits atomic and focused
- Write meaningful commit messages
- Review your own PR before requesting reviews
- Use draft PRs for work in progress
- Squash commits when merging if history is messy
- Never force push to shared branches
- Use `--force-with-lease` instead of `--force`
