---
name: algorithmic-art
description: Create generative art through computational aesthetics using p5.js, emphasizing seeded randomness, interactive parameter exploration, and reproducible algorithmic outputs.
---

# Algorithmic Art Skill

This skill enables creation of generative art through computational aesthetics using p5.js, emphasizing seeded randomness and interactive parameter exploration rather than static imagery.

## Two-Phase Process

### Phase 1: Philosophical Foundation

Before coding, create a 4-6 paragraph algorithmic manifesto articulating:

- **Computational processes** and mathematical relationships
- **Noise functions** and randomness patterns
- **Particle behaviors** and field dynamics
- **Temporal evolution** and system states
- **Parametric variation** and emergent complexity

This philosophy guides implementation decisions and ensures cohesive artistic vision.

### Phase 2: Code Implementation

Express the philosophy through a self-contained HTML artifact featuring:
- p5.js algorithms
- Parameter controls
- Seed navigation (prev/next/random/jump)
- Reproducible outputs

## Template Structure

```html
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Algorithmic Art</title>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/p5.js/1.9.0/p5.min.js"></script>
  <style>
    * { margin: 0; padding: 0; box-sizing: border-box; }
    body {
      display: flex;
      background: #1a1a1a;
      color: #ffffff;
      font-family: 'Courier New', monospace;
    }
    #canvas-container { flex: 1; display: flex; justify-content: center; align-items: center; }
    #controls {
      width: 280px;
      padding: 20px;
      background: #242424;
      overflow-y: auto;
      max-height: 100vh;
    }
    .control-group { margin-bottom: 20px; }
    .control-group label { display: block; margin-bottom: 5px; font-size: 12px; }
    .control-group input[type="range"] { width: 100%; }
    .seed-controls { display: flex; gap: 8px; margin-bottom: 20px; }
    .seed-controls button {
      flex: 1;
      padding: 8px;
      background: #3a3a3a;
      border: none;
      color: white;
      cursor: pointer;
    }
    .seed-controls button:hover { background: #4a4a4a; }
    #seed-display { text-align: center; margin-bottom: 10px; font-size: 14px; }
  </style>
</head>
<body>
  <div id="canvas-container"></div>
  <div id="controls">
    <div id="seed-display">Seed: <span id="current-seed">0</span></div>
    <div class="seed-controls">
      <button onclick="prevSeed()">Prev</button>
      <button onclick="nextSeed()">Next</button>
      <button onclick="randomSeed()">Random</button>
    </div>

    <!-- Parameters go here -->
    <div class="control-group">
      <label>Density: <span id="density-val">50</span></label>
      <input type="range" id="density" min="10" max="200" value="50"
             oninput="params.density = this.value; document.getElementById('density-val').textContent = this.value; regenerate();">
    </div>

    <button onclick="regenerate()" style="width:100%; padding:12px; margin-top:20px;">
      Regenerate
    </button>
    <button onclick="saveCanvas('art', 'png')" style="width:100%; padding:12px; margin-top:10px;">
      Download PNG
    </button>
  </div>

  <script>
    let seed = Math.floor(Math.random() * 100000);
    let params = {
      density: 50
    };

    function setup() {
      let canvas = createCanvas(800, 800);
      canvas.parent('canvas-container');
      noLoop();
      regenerate();
    }

    function regenerate() {
      randomSeed(seed);
      noiseSeed(seed);
      document.getElementById('current-seed').textContent = seed;

      // Your algorithm here
      background(20);

      for (let i = 0; i < params.density; i++) {
        let x = random(width);
        let y = random(height);
        let size = noise(x * 0.01, y * 0.01) * 100;

        noStroke();
        fill(255, 100);
        ellipse(x, y, size, size);
      }
    }

    function prevSeed() { seed--; regenerate(); }
    function nextSeed() { seed++; regenerate(); }
    function randomSeed() { seed = Math.floor(Math.random() * 100000); regenerate(); }
  </script>
</body>
</html>
```

## Core Techniques

### Seeded Randomness

```javascript
// Art Blocks pattern for reproducibility
randomSeed(seed);
noiseSeed(seed);

// Now all random() and noise() calls are deterministic
let x = random(width);  // Same result for same seed
let n = noise(x * 0.01);
```

### Perlin Noise Flow Fields

```javascript
function drawFlowField() {
  let scale = 0.005;
  let particles = [];

  // Initialize particles
  for (let i = 0; i < 1000; i++) {
    particles.push({
      x: random(width),
      y: random(height),
      prevX: 0,
      prevY: 0
    });
  }

  // Update and draw
  for (let p of particles) {
    p.prevX = p.x;
    p.prevY = p.y;

    let angle = noise(p.x * scale, p.y * scale) * TWO_PI * 2;
    p.x += cos(angle) * 2;
    p.y += sin(angle) * 2;

    stroke(255, 20);
    line(p.prevX, p.prevY, p.x, p.y);
  }
}
```

### Recursive Subdivision

```javascript
function subdivide(x, y, w, h, depth) {
  if (depth <= 0 || w < 10 || h < 10) {
    // Draw leaf
    noStroke();
    fill(random(255), random(255), random(255), 100);
    rect(x, y, w, h);
    return;
  }

  if (random() > 0.5) {
    // Horizontal split
    let split = random(0.3, 0.7) * h;
    subdivide(x, y, w, split, depth - 1);
    subdivide(x, y + split, w, h - split, depth - 1);
  } else {
    // Vertical split
    let split = random(0.3, 0.7) * w;
    subdivide(x, y, split, h, depth - 1);
    subdivide(x + split, y, w - split, h, depth - 1);
  }
}
```

### Particle Systems

```javascript
class Particle {
  constructor() {
    this.pos = createVector(random(width), random(height));
    this.vel = createVector(0, 0);
    this.acc = createVector(0, 0);
    this.maxSpeed = 4;
  }

  applyForce(force) {
    this.acc.add(force);
  }

  update() {
    this.vel.add(this.acc);
    this.vel.limit(this.maxSpeed);
    this.pos.add(this.vel);
    this.acc.mult(0);

    // Wrap around edges
    if (this.pos.x > width) this.pos.x = 0;
    if (this.pos.x < 0) this.pos.x = width;
    if (this.pos.y > height) this.pos.y = 0;
    if (this.pos.y < 0) this.pos.y = height;
  }

  draw() {
    stroke(255, 50);
    point(this.pos.x, this.pos.y);
  }
}
```

## Parameter Design

Parameters should emerge from algorithmic needs:

| Type | Examples |
|------|----------|
| Quantities | Particle count, subdivision depth |
| Scales | Noise scale, line weight, spacing |
| Probabilities | Branch chance, color variation |
| Ratios | Golden ratio, aspect proportions |
| Angles | Rotation step, flow direction |
| Thresholds | Density cutoffs, distance limits |

## Deliverable Format

Single, self-contained HTML artifact with:
- Embedded p5.js via CDN
- Inline CSS and JavaScript
- Zero external dependencies
- Seed-based reproducibility
- Interactive parameter controls
- Download functionality

## Tips

- Start with a clear conceptual direction
- Let algorithms reveal emergent behaviors
- Use noise for organic, natural variations
- Embrace controlled randomness
- Test many seed values to find interesting states
- Document your algorithm's "personality"
