---
name: pptx-toolkit
description: Create, edit, and analyze PowerPoint presentations (.pptx files) with support for templates, slide layouts, animations, and professional presentation workflows.
---

# PowerPoint Presentation Toolkit

This skill enables Claude to create, edit, and analyze PowerPoint presentations (.pptx files), which are essentially ZIP archives containing XML files and other resources.

## Core Libraries & Tools

| Tool | Best For |
|------|----------|
| **python-pptx** | Python presentation creation/editing |
| **pptxgenjs** | JavaScript/Node.js presentations |
| **html2pptx** | Convert HTML designs to slides |
| **pandoc** | Text extraction and conversion |

## Key Workflows

### 1. Reading & Analyzing Presentations

**Text Extraction**:
```python
from pptx import Presentation

prs = Presentation('presentation.pptx')

for slide_num, slide in enumerate(prs.slides, 1):
    print(f"\n--- Slide {slide_num} ---")
    for shape in slide.shapes:
        if shape.has_text_frame:
            for paragraph in shape.text_frame.paragraphs:
                print(paragraph.text)

        # Extract from tables
        if shape.has_table:
            for row in shape.table.rows:
                for cell in row.cells:
                    print(cell.text)
```

**Extract Speaker Notes**:
```python
for slide in prs.slides:
    if slide.has_notes_slide:
        notes = slide.notes_slide.notes_text_frame.text
        print(notes)
```

### 2. Creating Presentations (Python)

```python
from pptx import Presentation
from pptx.util import Inches, Pt
from pptx.enum.text import PP_ALIGN
from pptx.dml.color import RGBColor

prs = Presentation()

# Title slide
title_slide_layout = prs.slide_layouts[0]
slide = prs.slides.add_slide(title_slide_layout)

title = slide.shapes.title
subtitle = slide.placeholders[1]

title.text = "Presentation Title"
subtitle.text = "Subtitle here"

# Content slide
bullet_slide_layout = prs.slide_layouts[1]
slide = prs.slides.add_slide(bullet_slide_layout)

title = slide.shapes.title
title.text = "Key Points"

body = slide.placeholders[1]
tf = body.text_frame

tf.text = "First bullet point"
p = tf.add_paragraph()
p.text = "Second bullet point"
p.level = 0

p = tf.add_paragraph()
p.text = "Sub-bullet"
p.level = 1

prs.save('output.pptx')
```

### 3. Creating with JavaScript (pptxgenjs)

```javascript
import pptxgen from 'pptxgenjs';

const pptx = new pptxgen();

// Set presentation properties
pptx.author = 'Author Name';
pptx.title = 'Presentation Title';
pptx.subject = 'Subject';

// Title slide
let slide = pptx.addSlide();
slide.addText('Presentation Title', {
  x: '10%',
  y: '40%',
  w: '80%',
  h: '20%',
  fontSize: 44,
  bold: true,
  align: 'center',
  color: '363636'
});

// Content slide
slide = pptx.addSlide();
slide.addText('Key Points', {
  x: 0.5,
  y: 0.5,
  w: 9,
  h: 1,
  fontSize: 32,
  bold: true,
  color: '363636'
});

slide.addText([
  { text: 'First point', options: { bullet: true } },
  { text: 'Second point', options: { bullet: true } },
  { text: 'Third point', options: { bullet: true } }
], {
  x: 0.5,
  y: 1.5,
  w: 9,
  h: 4,
  fontSize: 18,
  color: '666666'
});

// Save
await pptx.writeFile({ fileName: 'output.pptx' });
```

## Slide Layouts

Standard layout indices (may vary by template):

| Index | Layout Name |
|-------|-------------|
| 0 | Title Slide |
| 1 | Title and Content |
| 2 | Section Header |
| 3 | Two Content |
| 4 | Comparison |
| 5 | Title Only |
| 6 | Blank |
| 7 | Content with Caption |
| 8 | Picture with Caption |

## Adding Visual Elements

### Images

```python
from pptx.util import Inches

slide = prs.slides.add_slide(prs.slide_layouts[6])  # Blank

# Add image with position and size
slide.shapes.add_picture(
    'image.png',
    Inches(1),      # left
    Inches(1),      # top
    Inches(5),      # width
    Inches(3.75)    # height
)
```

### Tables

```python
from pptx.util import Inches, Pt

slide = prs.slides.add_slide(prs.slide_layouts[6])

# Define table dimensions
rows, cols = 4, 3
left, top, width, height = Inches(1), Inches(2), Inches(8), Inches(3)

table = slide.shapes.add_table(rows, cols, left, top, width, height).table

# Set column widths
table.columns[0].width = Inches(3)
table.columns[1].width = Inches(3)
table.columns[2].width = Inches(2)

# Header row
headers = ['Name', 'Role', 'Status']
for i, header in enumerate(headers):
    cell = table.cell(0, i)
    cell.text = header
    cell.text_frame.paragraphs[0].font.bold = True

# Data rows
data = [
    ['Alice', 'Developer', 'Active'],
    ['Bob', 'Designer', 'Active'],
    ['Carol', 'Manager', 'Away'],
]

for row_idx, row_data in enumerate(data, 1):
    for col_idx, value in enumerate(row_data):
        table.cell(row_idx, col_idx).text = value
```

### Shapes

```python
from pptx.enum.shapes import MSO_SHAPE
from pptx.dml.color import RGBColor

slide = prs.slides.add_slide(prs.slide_layouts[6])

# Rectangle
shape = slide.shapes.add_shape(
    MSO_SHAPE.RECTANGLE,
    Inches(1), Inches(1),
    Inches(3), Inches(2)
)
shape.fill.solid()
shape.fill.fore_color.rgb = RGBColor(0x42, 0x72, 0xC4)

# Add text to shape
shape.text_frame.text = "Box with text"
shape.text_frame.paragraphs[0].font.color.rgb = RGBColor(0xFF, 0xFF, 0xFF)
```

### Charts

```python
from pptx.chart.data import CategoryChartData
from pptx.enum.chart import XL_CHART_TYPE

slide = prs.slides.add_slide(prs.slide_layouts[6])

chart_data = CategoryChartData()
chart_data.categories = ['Q1', 'Q2', 'Q3', 'Q4']
chart_data.add_series('Sales', (100, 150, 120, 180))

chart = slide.shapes.add_chart(
    XL_CHART_TYPE.COLUMN_CLUSTERED,
    Inches(1), Inches(1.5),
    Inches(8), Inches(5),
    chart_data
).chart

chart.has_legend = True
```

## Working with Templates

```python
from pptx import Presentation

# Load template
prs = Presentation('template.pptx')

# Access and modify existing slides
for slide in prs.slides:
    for shape in slide.shapes:
        if shape.has_text_frame:
            for paragraph in shape.text_frame.paragraphs:
                if '{{TITLE}}' in paragraph.text:
                    paragraph.text = paragraph.text.replace('{{TITLE}}', 'Actual Title')

# Add new slides using template layouts
slide_layout = prs.slide_layouts[1]
new_slide = prs.slides.add_slide(slide_layout)

prs.save('output_from_template.pptx')
```

## Design Best Practices

1. **Consistent styling** - Use master slides for fonts and colors
2. **Readable fonts** - Minimum 24pt for body text, 36pt for titles
3. **Limited text** - 6x6 rule (6 bullets, 6 words each)
4. **High contrast** - Ensure text is readable against backgrounds
5. **Quality images** - Use high-resolution images (at least 150 DPI)

## Tips

- PowerPoint files are ZIP archives - unzip to inspect XML
- Use pandoc for quick text extraction
- Test output in PowerPoint to verify rendering
- Keep presentations under 20 slides for best engagement
- Use speaker notes for additional context
- Export to PDF for universal sharing
