---
name: xlsx-toolkit
description: Comprehensive spreadsheet tool supporting formulas, formatting, data analysis, and visualization for .xlsx, .xlsm, .csv, and .tsv files with professional standards.
---

# Excel Spreadsheet Toolkit

Comprehensive spreadsheet tool supporting formulas, formatting, data analysis, and visualization for .xlsx, .xlsm, .csv, and .tsv files.

## Core Libraries

| Library | Best For |
|---------|----------|
| **pandas** | Data analysis, reading/writing, transformations |
| **openpyxl** | Complex formatting, formulas, charts |
| **xlsxwriter** | Creating new files with advanced features |

## Excel Output Standards

### Zero Formula Errors Required

Always check for and fix these errors:
- `#REF!` - Invalid cell reference
- `#DIV/0!` - Division by zero
- `#VALUE!` - Wrong value type
- `#N/A` - Value not available
- `#NAME?` - Unrecognized formula name

### Financial Model Color Coding

| Color | Meaning |
|-------|---------|
| **Blue text** | Hardcoded inputs, user-changeable values |
| **Black text** | Formulas and calculations |
| **Green text** | Internal worksheet links |
| **Red text** | External file links |
| **Yellow background** | Key assumptions requiring attention |

### Number Formatting Standards

```python
from openpyxl.styles import numbers

# Years as text
cell.number_format = '@'  # e.g., "2024"

# Currency with thousands separator
cell.number_format = '"$"#,##0.00'

# Zeros as dashes
cell.number_format = '#,##0;-#,##0;"-"'

# Percentages
cell.number_format = '0.0%'

# Negative numbers in parentheses
cell.number_format = '#,##0;(#,##0)'
```

## Critical Formula Rules

1. **Always use Excel formulas** instead of calculating in Python and hardcoding
2. **Place all assumptions** in separate cells with clear labels
3. **Use cell references** rather than hardcoded values
4. **Test 2-3 sample references** before building full models

## Reading Files

### Basic Reading with Pandas

```python
import pandas as pd

# Read Excel file
df = pd.read_excel('data.xlsx', sheet_name='Sheet1')

# Read specific columns
df = pd.read_excel('data.xlsx', usecols=['A', 'B', 'D'])

# Read with header on different row
df = pd.read_excel('data.xlsx', header=2)

# Read CSV/TSV
df = pd.read_csv('data.csv')
df = pd.read_csv('data.tsv', sep='\t')
```

### Reading with openpyxl (Preserve Formatting)

```python
from openpyxl import load_workbook

wb = load_workbook('data.xlsx', data_only=False)  # Keep formulas
ws = wb.active

# Access cell values
value = ws['A1'].value
formula = ws['B1'].value  # Returns formula string

# Iterate through rows
for row in ws.iter_rows(min_row=2, max_col=5, values_only=True):
    print(row)
```

## Creating Spreadsheets

### With openpyxl

```python
from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill, Alignment, Border, Side

wb = Workbook()
ws = wb.active
ws.title = "Report"

# Headers with styling
headers = ['Date', 'Product', 'Quantity', 'Price', 'Total']
for col, header in enumerate(headers, 1):
    cell = ws.cell(row=1, column=col, value=header)
    cell.font = Font(bold=True, color='FFFFFF')
    cell.fill = PatternFill(start_color='4472C4', fill_type='solid')
    cell.alignment = Alignment(horizontal='center')

# Data with formulas
data = [
    ['2024-01-01', 'Widget A', 10, 25.00],
    ['2024-01-02', 'Widget B', 5, 40.00],
    ['2024-01-03', 'Widget A', 15, 25.00],
]

for row_num, row_data in enumerate(data, 2):
    for col_num, value in enumerate(row_data, 1):
        ws.cell(row=row_num, column=col_num, value=value)

    # Add formula for Total (Quantity * Price)
    ws.cell(row=row_num, column=5, value=f'=C{row_num}*D{row_num}')

# Column widths
ws.column_dimensions['A'].width = 12
ws.column_dimensions['B'].width = 15

wb.save('report.xlsx')
```

### Financial Model Template

```python
from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill

wb = Workbook()
ws = wb.active
ws.title = "Model"

# Assumptions section (blue text for inputs)
ws['A1'] = "ASSUMPTIONS"
ws['A1'].font = Font(bold=True)

ws['A2'] = "Growth Rate"
ws['B2'] = 0.05
ws['B2'].font = Font(color='0000FF')  # Blue for inputs
ws['B2'].number_format = '0.0%'

ws['A3'] = "Tax Rate"
ws['B3'] = 0.25
ws['B3'].font = Font(color='0000FF')
ws['B3'].number_format = '0.0%'

# Calculations section (black text for formulas)
ws['A5'] = "CALCULATIONS"
ws['A5'].font = Font(bold=True)

ws['A6'] = "Base Revenue"
ws['B6'] = 1000000
ws['B6'].font = Font(color='0000FF')  # Input
ws['B6'].number_format = '"$"#,##0'

ws['A7'] = "Year 1 Revenue"
ws['B7'] = '=B6*(1+B2)'  # Formula references assumption
ws['B7'].number_format = '"$"#,##0'

ws['A8'] = "Year 1 Tax"
ws['B8'] = '=B7*B3'
ws['B8'].number_format = '"$"#,##0'

wb.save('financial_model.xlsx')
```

## Data Analysis

### Pivot Tables

```python
import pandas as pd

df = pd.read_excel('sales.xlsx')

# Create pivot table
pivot = pd.pivot_table(
    df,
    values='Revenue',
    index='Region',
    columns='Quarter',
    aggfunc='sum',
    margins=True
)

# Save to new sheet
with pd.ExcelWriter('analysis.xlsx') as writer:
    df.to_excel(writer, sheet_name='Raw Data', index=False)
    pivot.to_excel(writer, sheet_name='Pivot')
```

### Charts

```python
from openpyxl import Workbook
from openpyxl.chart import BarChart, Reference

wb = Workbook()
ws = wb.active

# Add data
data = [
    ['Month', 'Sales'],
    ['Jan', 1500],
    ['Feb', 2000],
    ['Mar', 1800],
    ['Apr', 2200],
]

for row in data:
    ws.append(row)

# Create chart
chart = BarChart()
chart.title = "Monthly Sales"
chart.x_axis.title = "Month"
chart.y_axis.title = "Sales ($)"

data_ref = Reference(ws, min_col=2, min_row=1, max_row=5)
cats_ref = Reference(ws, min_col=1, min_row=2, max_row=5)

chart.add_data(data_ref, titles_from_data=True)
chart.set_categories(cats_ref)

ws.add_chart(chart, "D2")

wb.save('chart_example.xlsx')
```

## Verification Checklist

Before finalizing any spreadsheet:

- [ ] Test 2-3 sample cell references
- [ ] Confirm column/row mapping accuracy
- [ ] Check for division by zero errors
- [ ] Verify all cell references exist
- [ ] Run formula recalculation
- [ ] Validate number formatting
- [ ] Check color coding consistency

## Tips

- Always use formulas over hardcoded calculations
- Document assumptions in separate cells
- Use named ranges for clarity
- Apply consistent formatting
- Test with sample data first
- Keep templates for common reports
