---
name: web-artifacts-builder
description: Create sophisticated multi-component web artifacts using React, TypeScript, Tailwind CSS, and shadcn/ui components bundled into self-contained HTML files.
---

# Web Artifacts Builder

This skill provides tools for creating sophisticated multi-component artifacts using modern frontend technologies including React, Tailwind CSS, and shadcn/ui components.

## Technology Stack

- **React 18** + TypeScript
- **Vite** for development
- **Parcel** for bundling
- **Tailwind CSS 3.4.1** with shadcn/ui theming
- **40+ shadcn/ui components** pre-installed
- **Path aliases** using `@/` notation

## Workflow

### 1. Initialize Project

```bash
# Create new artifact project
./scripts/init-artifact.sh my-artifact

# This creates:
# my-artifact/
# ├── index.html
# ├── package.json
# ├── vite.config.ts
# ├── tailwind.config.js
# ├── tsconfig.json
# └── src/
#     ├── App.tsx
#     ├── main.tsx
#     └── components/
```

### 2. Develop Your Artifact

```tsx
// src/App.tsx
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"

export default function App() {
  return (
    <div className="min-h-screen bg-background p-8">
      <Card className="max-w-md mx-auto">
        <CardHeader>
          <CardTitle>My Artifact</CardTitle>
        </CardHeader>
        <CardContent>
          <Button onClick={() => alert('Hello!')}>
            Click Me
          </Button>
        </CardContent>
      </Card>
    </div>
  )
}
```

### 3. Bundle for Distribution

```bash
# Bundle into single HTML file
./scripts/bundle-artifact.sh my-artifact

# Output: my-artifact/dist/index.html (self-contained)
```

### 4. Share the Artifact

The bundled HTML file is completely self-contained and can be:
- Shared directly in Claude conversations
- Opened in any browser
- Hosted anywhere

## Available shadcn/ui Components

```
Accordion      Alert          AlertDialog    Avatar
Badge          Button         Calendar       Card
Checkbox       Collapsible    Command        ContextMenu
Dialog         DropdownMenu   Form           HoverCard
Input          Label          Menubar        NavigationMenu
Popover        Progress       RadioGroup     ScrollArea
Select         Separator      Sheet          Skeleton
Slider         Switch         Table          Tabs
Textarea       Toast          Toggle         Tooltip
```

## Design Guidelines

### Avoid Generic AI Aesthetics

**Don't use:**
- Centered layouts for everything
- Purple/blue gradients
- Consistent rounded corners everywhere
- Inter typeface as default

**Do use:**
- Intentional, purposeful layouts
- Brand-appropriate color schemes
- Mixed border radius for visual interest
- Distinctive typography

### Example: Dashboard Layout

```tsx
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs"

export default function Dashboard() {
  return (
    <div className="min-h-screen bg-slate-50">
      {/* Asymmetric header */}
      <header className="bg-slate-900 text-white p-8">
        <div className="max-w-6xl mx-auto">
          <h1 className="text-4xl font-bold tracking-tight">Dashboard</h1>
          <p className="text-slate-400 mt-2">Analytics overview</p>
        </div>
      </header>

      {/* Content with varied card sizes */}
      <main className="max-w-6xl mx-auto p-8 -mt-8">
        <div className="grid grid-cols-3 gap-6">
          <Card className="col-span-2 shadow-lg">
            <CardHeader>
              <CardTitle>Revenue</CardTitle>
            </CardHeader>
            <CardContent>
              {/* Chart component */}
            </CardContent>
          </Card>

          <Card className="shadow-lg">
            <CardHeader>
              <CardTitle>Quick Stats</CardTitle>
            </CardHeader>
            <CardContent>
              {/* Stats list */}
            </CardContent>
          </Card>
        </div>
      </main>
    </div>
  )
}
```

## Bundling Requirements

For successful bundling, ensure:

1. **index.html exists** in project root
2. **Valid React entry point** in src/main.tsx
3. **No external runtime dependencies** (CDN links are fine)

```html
<!-- index.html -->
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>My Artifact</title>
</head>
<body>
  <div id="root"></div>
  <script type="module" src="/src/main.tsx"></script>
</body>
</html>
```

## Scripts Reference

### init-artifact.sh

```bash
#!/bin/bash
# Initialize new artifact project
PROJECT_NAME=$1

mkdir -p "$PROJECT_NAME/src/components/ui"

# Create package.json, configs, and starter files
# Installs dependencies automatically
```

### bundle-artifact.sh

```bash
#!/bin/bash
# Bundle artifact into single HTML
PROJECT_NAME=$1

cd "$PROJECT_NAME"
npm run build

# Use Parcel + html-inline to create self-contained file
npx parcel build index.html --no-source-maps
npx html-inline dist/index.html -o dist/bundled.html
```

## Tips

- Use CSS variables for theming consistency
- Leverage shadcn/ui's built-in dark mode support
- Keep bundle size reasonable by importing only needed components
- Test the bundled HTML in multiple browsers
- Consider accessibility from the start
