---
name: frontend-design
description: Create distinctive, production-grade frontend interfaces that prioritize authentic design over generic aesthetics, avoiding AI-generated design patterns.
---

# Frontend Design Skill

This skill guides creation of distinctive, production-grade frontend interfaces that prioritize authentic design over generic aesthetics. Generate creative, polished code and UI design that avoids generic AI aesthetics.

## Initial Analysis

Before writing any code, understand context by examining:
- **Purpose**: What is the core function?
- **Target Audience**: Who are the users?
- **Tone**: Professional, playful, bold, minimal?
- **Technical Constraints**: Framework, browser support, performance needs?
- **Differentiation**: What makes this unique?

## Aesthetic Direction

> "Choose a clear conceptual direction and execute it with precision. Bold maximalism and refined minimalism both work - the key is intentionality, not intensity."

Avoid the generic middle ground. Commit to a direction.

## Implementation Focus Areas

### 1. Typography

**Don't**: Default to Inter, Roboto, or system fonts
**Do**: Select distinctive fonts that reinforce the design direction

```css
/* Example: Editorial feel */
@import url('https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Source+Sans+Pro:wght@400;600&display=swap');

:root {
  --font-display: 'Playfair Display', serif;
  --font-body: 'Source Sans Pro', sans-serif;
}

h1, h2, h3 {
  font-family: var(--font-display);
  letter-spacing: -0.02em;
}
```

Pair a distinctive display font with a refined body font.

### 2. Color & Theme

**Don't**: Distribute colors timidly across the interface
**Do**: Use dominant colors with sharp accents

```css
:root {
  /* Bold primary with intentional accent */
  --color-primary: #1a1a2e;
  --color-accent: #e94560;
  --color-surface: #f8f9fa;
  --color-text: #16213e;

  /* Semantic colors */
  --color-success: #06d6a0;
  --color-warning: #ffd166;
  --color-error: #ef476f;
}

/* Dark mode with intention */
[data-theme="dark"] {
  --color-primary: #0f0f1a;
  --color-surface: #1a1a2e;
  --color-text: #edf2f4;
}
```

### 3. Motion & Interaction

**Don't**: Scatter micro-interactions everywhere
**Do**: Emphasize high-impact moments

```css
/* Staggered reveal animation */
@keyframes fadeInUp {
  from {
    opacity: 0;
    transform: translateY(20px);
  }
  to {
    opacity: 1;
    transform: translateY(0);
  }
}

.card {
  animation: fadeInUp 0.6s ease-out forwards;
  opacity: 0;
}

.card:nth-child(1) { animation-delay: 0.1s; }
.card:nth-child(2) { animation-delay: 0.2s; }
.card:nth-child(3) { animation-delay: 0.3s; }

/* Scroll-triggered effects */
.section {
  opacity: 0;
  transform: translateY(40px);
  transition: all 0.8s cubic-bezier(0.16, 1, 0.3, 1);
}

.section.visible {
  opacity: 1;
  transform: translateY(0);
}
```

### 4. Spatial Composition

**Don't**: Default to centered, symmetric layouts
**Do**: Use unexpected layouts, asymmetry, overlap, diagonal flow

```css
/* Asymmetric grid */
.hero-grid {
  display: grid;
  grid-template-columns: 1.5fr 1fr;
  gap: 4rem;
  align-items: end;
}

/* Overlapping elements */
.feature-card {
  position: relative;
}

.feature-card::before {
  content: '';
  position: absolute;
  top: -20px;
  left: -20px;
  width: 60%;
  height: 60%;
  background: var(--color-accent);
  opacity: 0.1;
  z-index: -1;
}

/* Breaking the grid */
.highlight-section {
  margin-left: -5vw;
  margin-right: -5vw;
  padding: 4rem 5vw;
}
```

### 5. Visual Details

Create atmosphere through textures and effects:

```css
/* Gradient mesh background */
.hero {
  background:
    radial-gradient(ellipse at 20% 80%, rgba(233, 69, 96, 0.15) 0%, transparent 50%),
    radial-gradient(ellipse at 80% 20%, rgba(6, 214, 160, 0.1) 0%, transparent 50%),
    var(--color-surface);
}

/* Noise texture overlay */
.textured {
  position: relative;
}

.textured::after {
  content: '';
  position: absolute;
  inset: 0;
  background-image: url("data:image/svg+xml,%3Csvg viewBox='0 0 256 256' xmlns='http://www.w3.org/2000/svg'%3E%3Cfilter id='noise'%3E%3CfeTurbulence type='fractalNoise' baseFrequency='0.8' numOctaves='4' stitchTiles='stitch'/%3E%3C/filter%3E%3Crect width='100%25' height='100%25' filter='url(%23noise)'/%3E%3C/svg%3E");
  opacity: 0.03;
  pointer-events: none;
}

/* Geometric accent */
.section-divider {
  height: 4px;
  background: linear-gradient(90deg, var(--color-accent), transparent);
  width: 120px;
}
```

## Anti-Patterns to Avoid

| Generic AI Pattern | Better Alternative |
|-------------------|-------------------|
| Purple/blue gradients | Intentional brand colors |
| Centered everything | Asymmetric, dynamic layouts |
| Rounded corners on all | Mix sharp and soft edges |
| Inter/Roboto fonts | Distinctive typography |
| Generic card grids | Varied content presentation |
| Floating blobs | Purposeful visual elements |

## Component Example

```tsx
// Distinctive button component
const Button = ({ children, variant = 'primary' }) => (
  <button
    className={`
      relative px-6 py-3 font-medium
      transition-all duration-300
      ${variant === 'primary'
        ? 'bg-accent text-white hover:translate-y-[-2px] hover:shadow-lg'
        : 'bg-transparent border-2 border-current hover:bg-accent hover:text-white hover:border-accent'
      }
      before:absolute before:inset-0 before:bg-white before:opacity-0
      hover:before:opacity-10 before:transition-opacity
    `}
  >
    {children}
  </button>
);
```

## Quality Checklist

- [ ] Typography is distinctive, not default
- [ ] Color palette has clear hierarchy
- [ ] Layout breaks expected patterns purposefully
- [ ] Motion emphasizes key interactions
- [ ] Visual details add atmosphere
- [ ] No generic AI aesthetic patterns
- [ ] Design serves the content's purpose
