---
name: skill-creator
description: Build modular skill packages that extend Claude's capabilities through specialized knowledge, workflows, and tool integrations following best practices.
---

# Skill Creator Guide

This skill provides guidance for building modular packages that extend Claude's capabilities through specialized knowledge, workflows, and tool integrations.

## Core Principles

### Concise Design
> "The context window is a public good."

Only include information Claude doesn't already possess. Challenge each element's necessity before including it. Remove redundant explanations, obvious instructions, and content Claude inherently knows.

### Appropriate Freedom Levels

Match specificity to task fragility:
- **High freedom**: Flexible approaches for creative or exploratory tasks
- **Low freedom**: Strict sequences for error-prone operations requiring exact steps

### Progressive Disclosure Pattern

Skills use three loading levels:
1. **Metadata** (always available): Name and description in frontmatter
2. **SKILL.md body** (when triggered): Main instructions loaded on activation
3. **Bundled resources** (as needed): Reference files loaded contextually

This manages context efficiency by keeping SKILL.md under 500 lines and deferring detailed content to reference files.

## Skill Structure

```
my-skill/
├── SKILL.md           # Required: Main instructions
├── scripts/           # Optional: Executable code
│   └── helper.py
├── references/        # Optional: Documentation loaded contextually
│   └── api-docs.md
└── assets/            # Optional: Templates, images, boilerplate
    └── template.html
```

### SKILL.md Format

```markdown
---
name: my-skill-name
description: Clear description of what this skill does and when to trigger it.
---

# My Skill Name

## When to Use
[Describe activation triggers]

## Workflow
[Step-by-step process]

## Guidelines
[Key rules and constraints]

## Examples
[Usage demonstrations]
```

### Component Purposes

| Component | Purpose | Context Loading |
|-----------|---------|-----------------|
| SKILL.md | Core instructions | On skill activation |
| scripts/ | Deterministic tasks, reusable code | Executed when needed |
| references/ | Detailed docs, specs | Loaded on demand |
| assets/ | Output templates, boilerplate | Not loaded, used in output |

## Creation Process

### 1. Understand the Skill
- Gather concrete usage examples
- Identify the core workflow
- Determine what Claude doesn't already know

### 2. Plan Reusable Contents
- What scripts can automate repetitive tasks?
- What references provide essential context?
- What assets support consistent output?

### 3. Initialize Structure

```bash
# Create skill directory
mkdir -p my-skill/{scripts,references,assets}

# Create SKILL.md with frontmatter
cat > my-skill/SKILL.md << 'EOF'
---
name: my-skill
description: Description here
---

# My Skill

Instructions here...
EOF
```

### 4. Write SKILL.md Content

**Keep under 500 lines.** Include:
- Clear trigger conditions
- Step-by-step workflow
- Essential guidelines only
- Reference file pointers for details

### 5. Add Supporting Files

**Scripts**: For deterministic, repeatable operations
```python
# scripts/validate.py
def validate_output(content):
    """Validation logic here"""
    pass
```

**References**: For detailed documentation
```markdown
# references/api-spec.md
Detailed API documentation that's too long for SKILL.md...
```

**Assets**: For output templates
```html
<!-- assets/template.html -->
<!DOCTYPE html>
<html>...</html>
```

### 6. Package and Validate

```bash
# Create ZIP package
cd my-skill
zip -r ../my-skill.zip .

# Validate structure
unzip -l ../my-skill.zip
```

## Best Practices

### Do
- Start with real usage examples
- Keep SKILL.md focused and concise
- Use references for detailed documentation
- Test with actual Claude interactions
- Iterate based on real-world usage

### Don't
- Include information Claude already knows
- Duplicate content between files
- Make SKILL.md too long (>500 lines)
- Add unnecessary complexity
- Forget to validate the package

## Quality Checklist

- [ ] SKILL.md has valid frontmatter (name, description)
- [ ] Description clearly states when to use this skill
- [ ] Instructions are concise and actionable
- [ ] References are used for detailed content
- [ ] Scripts are self-contained and documented
- [ ] Package validates without errors
- [ ] Tested with real Claude interactions

## Example: API Integration Skill

```markdown
---
name: stripe-integration
description: Integrate Stripe payment processing into web applications with proper error handling and webhook setup.
---

# Stripe Integration Skill

## When to Use
Activate when user needs to:
- Add payment processing to their app
- Set up Stripe webhooks
- Handle subscription billing

## Workflow
1. Check for existing Stripe setup
2. Install required packages
3. Configure environment variables
4. Implement payment flow
5. Set up webhook handling
6. Test integration

## Key Files
- `references/stripe-api.md`: Full API documentation
- `scripts/webhook-test.py`: Local webhook testing
- `assets/checkout-template.tsx`: React checkout component

## Guidelines
- Always use Stripe's official SDK
- Store API keys in environment variables
- Implement idempotency keys for payments
- Log all payment events for debugging
```
