---
name: webapp-testing
description: Test local web applications through native Python Playwright scripts with helper utilities managing server lifecycles for automated browser testing.
---

# Web Application Testing Skill

This toolkit enables testing of local web applications through native Python Playwright scripts, with helper utilities managing server lifecycles.

## Core Workflow

### Decision Tree for Task Selection

1. **Static HTML**: Read files directly to locate selectors, then compose Playwright scripts
2. **Dynamic Applications**: Determine if servers are operational, then follow reconnaissance-then-action sequence

### Essential Pattern for Dynamic Apps

```python
from playwright.sync_api import sync_playwright

with sync_playwright() as p:
    browser = p.chromium.launch(headless=False)
    page = browser.new_page()

    page.goto("http://localhost:3000")

    # CRITICAL: Wait for JS to execute before inspecting DOM
    page.wait_for_load_state('networkidle')

    # Now safe to inspect rendered content
    content = page.content()

    browser.close()
```

## Implementation Best Practices

### Selector Strategy (Priority Order)

1. **Role-based** (most resilient): `page.get_by_role("button", name="Submit")`
2. **Text-based**: `page.get_by_text("Login")`
3. **Test ID**: `page.get_by_test_id("submit-button")`
4. **CSS/ID**: `page.locator("#submit-btn")` or `page.locator(".btn-primary")`

### Wait Strategies

```python
# Wait for network to be idle
page.wait_for_load_state('networkidle')

# Wait for specific element
page.wait_for_selector('.loaded-content')

# Wait for element to be visible
page.locator('.modal').wait_for(state='visible')

# Custom timeout
page.wait_for_selector('.slow-content', timeout=10000)
```

### Screenshot and Visual Testing

```python
# Full page screenshot
page.screenshot(path="screenshot.png", full_page=True)

# Element screenshot
page.locator(".hero-section").screenshot(path="hero.png")

# Screenshot on failure
try:
    page.click("#missing-button")
except:
    page.screenshot(path="error-state.png")
    raise
```

### Form Interaction

```python
# Fill text inputs
page.fill('input[name="email"]', 'test@example.com')
page.fill('input[name="password"]', 'securepassword')

# Select dropdowns
page.select_option('select#country', 'US')

# Check/uncheck
page.check('input[type="checkbox"]')
page.uncheck('input#newsletter')

# Submit form
page.click('button[type="submit"]')
```

### Server Management Script

```python
# scripts/with_server.py - Manages server lifecycle
import subprocess
import time
import signal

def with_server(command, port, test_func):
    """Run test with managed server instance."""
    proc = subprocess.Popen(command, shell=True)

    # Wait for server to be ready
    for _ in range(30):
        try:
            import requests
            requests.get(f"http://localhost:{port}")
            break
        except:
            time.sleep(0.5)

    try:
        test_func()
    finally:
        proc.send_signal(signal.SIGTERM)
        proc.wait()
```

## Common Testing Scenarios

### Authentication Flow

```python
def test_login():
    with sync_playwright() as p:
        browser = p.chromium.launch()
        page = browser.new_page()

        page.goto("http://localhost:3000/login")
        page.fill('[name="email"]', 'user@test.com')
        page.fill('[name="password"]', 'password123')
        page.click('button:has-text("Sign In")')

        # Verify redirect to dashboard
        page.wait_for_url("**/dashboard")
        assert page.url.endswith("/dashboard")

        browser.close()
```

### API Response Validation

```python
def test_api_data_display():
    with sync_playwright() as p:
        browser = p.chromium.launch()
        page = browser.new_page()

        # Intercept API calls
        def handle_route(route):
            route.fulfill(json={"items": [{"id": 1, "name": "Test"}]})

        page.route("**/api/items", handle_route)
        page.goto("http://localhost:3000/items")

        # Verify mocked data is displayed
        assert page.get_by_text("Test").is_visible()

        browser.close()
```

### Console Error Detection

```python
def test_no_console_errors():
    errors = []

    with sync_playwright() as p:
        browser = p.chromium.launch()
        page = browser.new_page()

        page.on("console", lambda msg: errors.append(msg.text) if msg.type == "error" else None)

        page.goto("http://localhost:3000")
        page.wait_for_load_state('networkidle')

        assert len(errors) == 0, f"Console errors found: {errors}"

        browser.close()
```

## Tips

- Always close browsers after tests
- Use `headless=False` during development for visual debugging
- Leverage Playwright's auto-waiting where possible
- Use semantic selectors for resilience against UI changes
- Capture screenshots on failures for debugging
