---
name: mcp-builder
description: Create Model Context Protocol (MCP) servers that allow large language models to interact with external services through thoughtfully designed tools.
---

# MCP Server Development Guide

This skill enables developers to create Model Context Protocol (MCP) servers that allow large language models to interact with external services through thoughtfully designed tools.

## Four-Phase Development Process

### Phase 1: Deep Research and Planning

**Modern MCP Design Principles:**
- Balance comprehensive API endpoint coverage against specialized workflow tools
- Use consistent, action-oriented naming conventions with prefixes (e.g., `github_create_issue`)
- Design tools returning focused, relevant data with filtering/pagination support
- Craft error messages that guide agents toward solutions with specific next steps

**Documentation Review:**
Start with the MCP specification sitemap at `https://modelcontextprotocol.io/sitemap.xml` and review key architecture pages in markdown format.

**Recommended Technology Stack:**
- **Language:** TypeScript (preferred for SDK quality and broad AI model familiarity)
- **Transport:** Streamable HTTP for remote servers; stdio for local deployment

**API Planning:**
Analyze the target service's documentation, identify authentication requirements, and prioritize endpoints based on common usage patterns.

### Phase 2: Implementation

**Project Setup:**
Language-specific guides cover directory structure, configuration files, and dependency management.

**Core Infrastructure:**
Establish reusable components including:
- API clients with authentication
- Error handling utilities
- Response formatters
- Pagination support

**Tool Development Includes:**
- Input/output schemas using Zod (TypeScript) or Pydantic (Python)
- Clear parameter descriptions with usage examples
- Structured response content where possible
- Concise functionality summaries
- Proper async/await patterns for I/O operations
- Annotations indicating read-only, destructive, idempotent, or open-world behavior

### Phase 3: Review and Testing

**Quality Checks:**
- Eliminate code duplication
- Ensure consistent error handling
- Maintain complete type coverage
- Verify clear tool descriptions

**Build Verification:**
- TypeScript: `npm run build` compilation
- Python: `python -m py_compile` syntax validation
- Both: MCP Inspector testing via `npx @modelcontextprotocol/inspector`

### Phase 4: Creating Evaluations

**Evaluation Purpose:**
Test whether language models can effectively use the server to answer realistic, complex questions requiring multiple tool interactions.

**Evaluation Requirements:**
- 10 independent, complex questions
- Non-destructive operations only
- Real-world use cases
- Single, verifiable answers
- Stable answers over time

**Output Format:**
Questions and answers are structured in XML format with `<qa_pair>` elements containing `<question>` and `<answer>` tags.

## Example MCP Server Structure (TypeScript)

```typescript
import { Server } from "@modelcontextprotocol/sdk/server/index.js";
import { StdioServerTransport } from "@modelcontextprotocol/sdk/server/stdio.js";

const server = new Server(
  { name: "my-mcp-server", version: "1.0.0" },
  { capabilities: { tools: {} } }
);

server.setRequestHandler(ListToolsRequestSchema, async () => ({
  tools: [
    {
      name: "my_tool",
      description: "Description of what this tool does",
      inputSchema: {
        type: "object",
        properties: {
          param1: { type: "string", description: "Parameter description" }
        },
        required: ["param1"]
      }
    }
  ]
}));

const transport = new StdioServerTransport();
await server.connect(transport);
```

## Best Practices

1. **Naming Conventions**: Use `service_action_target` format (e.g., `github_create_issue`)
2. **Error Messages**: Include actionable guidance, not just error codes
3. **Pagination**: Support cursor-based pagination for list operations
4. **Rate Limiting**: Implement backoff strategies for API limits
5. **Authentication**: Store credentials securely, support multiple auth methods

## Resources

- [MCP Specification](https://modelcontextprotocol.io)
- [TypeScript SDK](https://github.com/modelcontextprotocol/typescript-sdk)
- [Python SDK](https://github.com/modelcontextprotocol/python-sdk)
