---
name: version-control
description: "Git version control with commit guidance, branching, conflict resolution, and releases. Use for: (1) Creating commits, (2) Managing branches (feature/*, bugfix/*), (3) Resolving conflicts, (4) Releases and tags."
---

# Version Control Skill

Git workflows, commit guidelines, and automated quality checks.

## Creating Commits

### Pre-Commit Workflow
1. **Check status**: `git status`
2. **Review style**: `git log --oneline -5`
3. **Stage files**: `git add <files>`
4. **Commit**: Write clear message (see guidelines below)

**Important**: Do NOT run `git diff` before committing - it consumes excessive tokens. Use `git status` output and your knowledge of changes.

### Commit Message Guidelines

**Rules**:
- Minimum 3 words
- Imperative mood ("Add feature" not "Added feature")
- Subject: 50-72 characters max
- Description: Max 7 lines, focus on WHAT and WHY

**Structure**:
```
Single line summary

- Key change 1
- Key change 2
- Key change 3
```

**Good Examples**:
```
Add user authentication feature
```

```
Fix null reference in payment module

- Add null check before processing payment
- Update error handling for edge cases
- Prevent crash when user data is incomplete
```

**Bad Examples**:
- "Fixed bug" (too vague)
- "Update" (too short)
- Too many implementation details

## Branching

**Feature branches**: `git checkout -b feature/<name>`
**Bug fixes**: `git checkout -b bugfix/<name>`

See [branching.md](references/branching.md) for complete conventions.

## Conflict Resolution

See [conflicts.md](references/conflicts.md) for detailed guidance.

**Quick commands**:
```bash
# Accept yours
git checkout --ours <file>

# Accept theirs
git checkout --theirs <file>

# Abort merge
git merge --abort
```

## Releases

See [releases.md](references/releases.md) for complete workflow.

**Quick release**:
```bash
git checkout -b release/v1.2.0
git commit -m "Bump version to v1.2.0"
git checkout master
git merge release/v1.2.0
git tag -a v1.2.0 -m "Release version 1.2.0"
git push origin master --tags
```

## Git Hooks

### Pre-commit Hook
Prevents commits with console/print statements.

**Setup**: `cp scripts/pre-commit .git/hooks/pre-commit && chmod +x .git/hooks/pre-commit`
**Bypass**: `git commit --no-verify`

### Commit-msg Hook
Validates commit messages (min 3 words).

**Setup**: `cp scripts/commit-msg .git/hooks/commit-msg && chmod +x .git/hooks/commit-msg`

### Post-merge Hook
Runs linter after merge.

**Setup**: `cp scripts/post-merge .git/hooks/post-merge && chmod +x .git/hooks/post-merge`

## Common Commands

```bash
# Amend last commit
git commit --amend

# Undo commit (keep changes)
git reset --soft HEAD~1

# Undo commit (discard changes)
git reset --hard HEAD~1

# Stash changes
git stash
git stash pop

# View history
git log --oneline --graph --all

# Create tag
git tag -a v1.2.0 -m "Release version 1.2.0"
```
