# Releases and Semantic Versioning

## Semantic Versioning (SemVer)

Format: **MAJOR.MINOR.PATCH** (e.g., `v1.2.3`)

### Version Components

- **MAJOR** (1.x.x): Breaking changes, incompatible API changes
- **MINOR** (x.2.x): New features, backward-compatible functionality
- **PATCH** (x.x.3): Bug fixes, backward-compatible fixes

### Pre-release Versions
- **Alpha**: `v1.2.3-alpha.1` - Early testing, unstable
- **Beta**: `v1.2.3-beta.1` - Feature complete, testing phase
- **RC**: `v1.2.3-rc.1` - Release candidate, final testing

## Release Workflow

### 1. Prepare Release Branch
```bash
# Create release branch from dev
git checkout dev
git pull origin dev
git checkout -b release/v1.2.0
```

### 2. Update Version Numbers
- Update version in `package.json`, `csproj`, `setup.py`, etc.
- Update changelog with release notes
- Commit version bump: `git commit -m "Bump version to v1.2.0"`

### 3. Testing and Fixes
- Run full test suite
- Fix any bugs found (small commits on release branch)
- No new features on release branch

### 4. Merge to Stage
```bash
git checkout stage
git merge release/v1.2.0
git push origin stage
# Deploy to staging environment
# Final QA testing
```

### 5. Merge to Master
```bash
git checkout master
git merge release/v1.2.0
git push origin master
```

### 6. Create Git Tag
```bash
# Create annotated tag
git tag -a v1.2.0 -m "Release version 1.2.0"

# Push tag to remote
git push origin v1.2.0

# Or push all tags
git push origin --tags
```

### 7. Merge Back to Dev
```bash
git checkout dev
git merge release/v1.2.0
git push origin dev
```

### 8. Delete Release Branch (Optional)
```bash
git branch -d release/v1.2.0
git push origin --delete release/v1.2.0
```

## Tag Management

### List Tags
```bash
# List all tags
git tag

# List tags matching pattern
git tag -l "v1.2.*"

# Show tag details
git show v1.2.0
```

### Create Tags

**Lightweight Tag** (simple pointer):
```bash
git tag v1.2.0
```

**Annotated Tag** (recommended, includes metadata):
```bash
git tag -a v1.2.0 -m "Release version 1.2.0"
```

**Tag Specific Commit**:
```bash
git tag -a v1.2.0 <commit-hash> -m "Release version 1.2.0"
```

### Delete Tags
```bash
# Delete local tag
git tag -d v1.2.0

# Delete remote tag
git push origin --delete v1.2.0
```

### Checkout Tag
```bash
# View code at specific tag
git checkout v1.2.0

# Create branch from tag
git checkout -b hotfix/v1.2.1 v1.2.0
```

## Changelog Template

```markdown
# Changelog

## [1.2.0] - 2024-11-28

### Added
- New feature: User authentication system
- API endpoint for password reset

### Changed
- Updated dashboard UI layout
- Improved database query performance

### Fixed
- Fixed login validation bug
- Resolved null reference in payment module

### Security
- Patched SQL injection vulnerability

## [1.1.0] - 2024-10-15
...
```

## Version Bump Decisions

### When to Bump MAJOR (x.0.0)
- Breaking API changes
- Removed features or endpoints
- Changed data structures/schemas
- Requires migration for users

### When to Bump MINOR (0.x.0)
- New features added
- New API endpoints
- Performance improvements
- Backward-compatible changes

### When to Bump PATCH (0.0.x)
- Bug fixes
- Security patches
- Documentation updates
- No new features

## Hotfix Process

For critical production bugs:

```bash
# Create hotfix from master
git checkout master
git checkout -b hotfix/v1.2.1

# Fix the bug
git commit -m "Fix critical payment processing bug"

# Bump patch version
# Update changelog

# Merge to master
git checkout master
git merge hotfix/v1.2.1
git tag -a v1.2.1 -m "Hotfix: Critical payment bug"
git push origin master --tags

# Merge to stage and dev
git checkout stage
git merge hotfix/v1.2.1
git checkout dev
git merge hotfix/v1.2.1

# Delete hotfix branch
git branch -d hotfix/v1.2.1
```
