# Conflict Resolution Guide

## Understanding Conflicts

Conflicts occur when Git cannot automatically merge changes because the same lines were modified differently in different branches.

## Conflict Markers

```
<<<<<<< HEAD (Current Change - your branch)
Your changes here
=======
Their changes here
>>>>>>> feature/other-branch (Incoming Change)
```

## Resolution Strategies

### 1. Accept Current (Keep Yours)
Use when your changes are correct and should be kept.

**Command**:
```bash
git checkout --ours <file>
git add <file>
```

### 2. Accept Incoming (Keep Theirs)
Use when incoming changes are correct and should be kept.

**Command**:
```bash
git checkout --theirs <file>
git add <file>
```

### 3. Manual Resolution (Most Common)
Use when both changes are important and need to be combined.

**Steps**:
1. Open the conflicted file in your editor
2. Look for conflict markers (`<<<<<<<`, `=======`, `>>>>>>>`)
3. Edit the file to combine both changes appropriately
4. Remove all conflict markers
5. Save the file
6. Stage the resolved file: `git add <file>`

### 4. Use Merge Tool
Visual tools can help with complex conflicts.

**Command**:
```bash
git mergetool
```

## Step-by-Step Resolution Process

### 1. Identify Conflicts
```bash
git status
# Look for files marked as "both modified"
```

### 2. Examine Each Conflict
```bash
git diff <file>
# Shows the conflicting sections
```

### 3. Choose Resolution Strategy
- Simple conflicts: Accept one side (`--ours` or `--theirs`)
- Complex conflicts: Manual edit
- Multiple conflicts: Use merge tool

### 4. Verify Resolution
```bash
# Check that no conflict markers remain
grep -r "<<<<<<" .
grep -r ">>>>>>>" .

# Ensure code still works
# Run tests, compile, etc.
```

### 5. Complete the Merge
```bash
git add <resolved-files>
git commit  # Or git merge --continue / git rebase --continue
```

## Common Conflict Scenarios

### Scenario 1: Different Code in Same Function
**Best Practice**: Manually merge, test thoroughly, ensure both features work together.

### Scenario 2: Deleted vs Modified File
**Resolution**: Decide if file should exist. If yes, restore and apply modifications. If no, confirm deletion.

### Scenario 3: Renamed File Conflicts
**Resolution**: Choose final name, update all references, ensure imports/paths are correct.

### Scenario 4: Configuration File Conflicts
**Best Practice**: Merge both configurations, ensure no duplicate keys, validate config syntax.

## Abort Operations

If resolution seems too complex:

**Abort Merge**:
```bash
git merge --abort
```

**Abort Rebase**:
```bash
git rebase --abort
```

**Abort Cherry-pick**:
```bash
git cherry-pick --abort
```

## Prevention Tips

1. **Pull regularly**: `git pull origin dev` to stay updated
2. **Small, frequent commits**: Easier to resolve conflicts
3. **Communicate**: Coordinate with team on overlapping work
4. **Merge dev into feature**: Keep feature branches updated with latest dev changes
