---
name: dotnet-backend-guidelines
description: Comprehensive backend development guide for .NET Core 8 / ASP.NET Core APIs. Use when creating controllers, services, repositories, middleware, or working with ASP.NET Core APIs, Entity Framework Core, dependency injection, validation, error handling, and clean architecture patterns. Covers layered architecture (Controllers → Services → Repositories), SOLID principles, async patterns, and best practices.
---

# .NET Core 8 Backend Development Guidelines

## Purpose

Establish consistency and best practices for .NET Core 8 / ASP.NET Core backend development using modern C# patterns, clean architecture, and SOLID principles.

## When to Use This Skill

Automatically activates when working on:
- Creating or modifying controllers, endpoints, APIs
- Building services, repositories
- Implementing middleware (auth, validation, error handling)
- Database operations with Entity Framework Core
- Dependency injection configuration
- Input validation
- Backend testing and refactoring

## ⚠️ CRITICAL: How to Use This Skill Properly

When this skill activates:
1. **Read this main file first** for overview and core principles
2. **Check the Navigation Guide table below** - it tells you which resource file to read based on your task
3. **ALWAYS read the recommended resource file** - it contains detailed patterns and examples
4. **Apply the patterns from BOTH files** - main file + resource file

**Example:**
- Task: "Create endpoint to filter employees by rank and assignment"
- Step 1: ✅ Read this main skill.md (you're doing this)
- Step 2: ✅ Check Navigation Guide → "Database access" → entity-framework-patterns.md
- Step 3: ✅ **READ entity-framework-patterns.md** before writing code
- Step 4: ✅ Apply query optimization patterns from that resource

---

## Quick Start

### New Backend Feature Checklist

- [ ] **Controller**: Minimal logic, delegate to services
- [ ] **Service**: Business logic with dependency injection
- [ ] **Repository**: Data access (if using repository pattern)
- [ ] **DTOs**: Data transfer objects for API contracts
- [ ] **MappingProfile*: map entity to its DTOs 
- [ ] **Validation**: FluentValidation or Data Annotations
- [ ] **Error Handling**: Global exception handling
- [ ] **Cache Policy**: If using [OutputCache], add policy to CachePolicyConfiguration.cs
- [ ] **Tests**: Unit + integration tests
- [ ] **DI Registration**: Register services in Program.cs

### New ASP.NET Core API Checklist

- [ ] Project structure (Controllers, Services, Data, Models, DTOs)
- [ ] Entity Framework Core DbContext
- [ ] Dependency injection configuration
- [ ] Global exception handling middleware
- [ ] CORS policy (if needed)
- [ ] Authentication/Authorization
- [ ] Swagger/OpenAPI documentation
- [ ] Logging configuration

---

## Architecture Overview

### Clean Architecture (4-Layer Pattern)

```
HTTP Request
    ↓
API Layer (YourProject.API)
    ↓ Controllers, Middleware, Program.cs
Application Layer (YourProject.Application)
    ↓ Services, DTOs, Mapping, Validators
Infrastructure Layer (YourProject.Infrastructure)
    ↓ Repositories, DbContext, Migrations
Domain Layer (YourProject.Core)
    ↓ Entities, Interfaces, Enums, Models
```

**Key Principle:** Dependencies point INWARD. Core has NO dependencies.

---

## Project Structure (Clean Architecture)

### Multi-Project Solution

```
YourSolution/
├── YourProject.API/                # Presentation Layer
│   ├── Controllers/V1/            # API controllers
│   ├── Middleware/                # Custom middleware
│   ├── Extensions/                # DI configuration extensions
│   ├── Utils/                     # API utilities
│   ├── Models/                    # API-specific models (ApiResponse)
│   ├── Program.cs                 # Application entry point
│   └── appsettings.json           # Configuration
│
├── YourProject.Application/        # Application Layer
│   ├── Services/                  # Business logic services
│   ├── Interfaces/                # Service contracts
│   ├── DTOs/                      # Data transfer objects
│   ├── Mapping/                   # AutoMapper profiles
│   ├── Validators/                # FluentValidation validators
│   └── EventHandlers/             # MediatR event handlers
│
├── YourProject.Infrastructure/     # Infrastructure Layer
│   ├── Data/                      # EF Core DbContext
│   ├── Repositories/              # Repository implementations
│   ├── Migrations/                # EF Core migrations
│   ├── Services/                  # External service integrations
│   └── Helpers/                   # Infrastructure utilities
│
└── YourProject.Core/               # Domain Layer
    ├── Entities/                  # Domain entities
    ├── Interfaces/                # Repository interfaces
    ├── Enums/                     # Business enums
    ├── Models/                    # Domain models (Filters, etc.)
    └── Extensions/                # Query extensions
```

**Naming Conventions:**
- Controllers: `PascalCase + Controller` - `ProductController.cs`
- Services: `PascalCase + Service` - `ProductService.cs`
- Repositories: `PascalCase + Repository` - `ProductRepository.cs`
- Interfaces: `I + PascalCase` - `IProductService.cs`, `IProductRepository.cs`
- Entities: `PascalCase` - `Product.cs`, `Order.cs`
- DTOs: `PascalCase + Dto` - `CreateProductDto.cs`, `ProductDetailsDto.cs`

---

## Core Principles (8 Key Rules)

### 1. Controllers Only Handle HTTP, Services Contain Logic

```csharp
// ❌ NEVER: Business logic in controllers
[HttpPost]
public async Task<IActionResult> CreateUser(CreateUserDto dto)
{
    // 200 lines of business logic
}

// ✅ ALWAYS: Delegate to service
[HttpPost]
public async Task<IActionResult> CreateUser(CreateUserDto dto)
{
    var user = await _userService.CreateUserAsync(dto);
    return CreatedAtAction(nameof(GetUser), new { id = user.Id }, user);
}
```

### 2. Use Dependency Injection

```csharp
public class UserController : ControllerBase
{
    private readonly IUserService _userService;
    private readonly ILogger<UserController> _logger;

    public UserController(IUserService userService, ILogger<UserController> logger)
    {
        _userService = userService;
        _logger = logger;
    }
}
```

### 3. Always Use Async/Await

```csharp
// ✅ ALWAYS async for I/O operations
public async Task<User> GetUserAsync(int id)
{
    return await _context.Users.FindAsync(id);
}

// ❌ NEVER block with .Result or .Wait()
```

### 4. Validate All Input

```csharp
// Using Data Annotations
public class CreateUserDto
{
    [Required]
    [StringLength(100)]
    public string Name { get; set; }

    [Required]
    [EmailAddress]
    public string Email { get; set; }
}

// Or FluentValidation (recommended)
public class CreateUserDtoValidator : AbstractValidator<CreateUserDto>
{
    public CreateUserDtoValidator()
    {
        RuleFor(x => x.Name).NotEmpty().MaximumLength(100);
        RuleFor(x => x.Email).NotEmpty().EmailAddress();
    }
}
```

### 5. Use Repository Pattern for Data Access

```csharp
// Service → Repository → Database
var users = await _userRepository.GetActiveUsersAsync();
```

### 6. Optimize Database Queries - Filter at Database Level

```csharp
// ❌ NEVER: Load all data then filter in memory
var employees = await _context.Employees
    .Include(e => e.Ranks)
    .Include(e => e.Assignments)
    .ToListAsync(); // Loads ALL employees!
var filtered = employees.Where(e => e.Ranks.Any(r => r.IsActive)).ToList();

// ✅ ALWAYS: Push filtering to database with LINQ
var filtered = await _context.Employees
    .Where(e => e.Ranks.Any(r => r.IsActive))
    .ToListAsync(); // Database filters before loading
```

**Key Performance Rules:**
- Use `.Where()`, `.Any()`, `.Select()` BEFORE `.ToListAsync()` or `.ToArrayAsync()`
- Avoid loading entire tables into memory
- Use `.Select()` to project only needed columns
- Use `.AsNoTracking()` for read-only queries
- Avoid N+1 query problems with proper `.Include()`

### 7. Global Exception Handling

```csharp
// Middleware for exception handling
public class ExceptionHandlingMiddleware
{
    private readonly RequestDelegate _next;
    private readonly ILogger<ExceptionHandlingMiddleware> _logger;

    public ExceptionHandlingMiddleware(RequestDelegate next,
        ILogger<ExceptionHandlingMiddleware> logger)
    {
        _next = next;
        _logger = logger;
    }

    public async Task InvokeAsync(HttpContext context)
    {
        try
        {
            await _next(context);
        }
        catch (Exception ex)
        {
            await HandleExceptionAsync(context, ex);
        }
    }
}
```

### 8. Follow SOLID Principles

- **S**ingle Responsibility
- **O**pen/Closed
- **L**iskov Substitution
- **I**nterface Segregation
- **D**ependency Inversion

### 9. Minimize Comments - Write Self-Documenting Code

Use clear naming and structure instead of comments. Only comment to explain **WHY**, not **WHAT**.

```csharp
// ❌ Avoid obvious comments
// Get user by ID
var user = await _repository.GetById(id);

// ✅ Only comment for business rules or non-obvious decisions
// Soft delete to preserve audit trail for compliance
user.IsDeleted = true;
```

**Comment Guidelines:**
- ✅ Use step comments (Step 1, Step 2) **only for complex or large multi-step processes**
- ❌ Don't use step comments for simple, straightforward code
- ❌ Don't repeat what the code already says
- ✅ Only explain complex business rules or WHY decisions were made

---

## Common Usings

```csharp
// ASP.NET Core
using Microsoft.AspNetCore.Mvc;
using Microsoft.AspNetCore.Authorization;

// Entity Framework Core
using Microsoft.EntityFrameworkCore;

// Dependency Injection
using Microsoft.Extensions.DependencyInjection;

// Logging
using Microsoft.Extensions.Logging;

// Validation
using FluentValidation;
using System.ComponentModel.DataAnnotations;

// Async
using System.Threading.Tasks;
```

---

## Quick Reference

### HTTP Status Codes

| Code | Use Case | ASP.NET Method |
|------|----------|----------------|
| 200 | Success | Ok() |
| 201 | Created | CreatedAtAction() |
| 204 | No Content | NoContent() |
| 400 | Bad Request | BadRequest() |
| 401 | Unauthorized | Unauthorized() |
| 403 | Forbidden | Forbid() |
| 404 | Not Found | NotFound() |
| 500 | Server Error | StatusCode(500) |

---

## Anti-Patterns to Avoid

❌ Business logic in controllers
❌ Blocking async calls with .Result or .Wait()
❌ Missing error handling
❌ No input validation
❌ Direct DbContext usage everywhere
❌ Console.WriteLine instead of ILogger
❌ Loading entire tables then filtering in memory (use LINQ Where before ToListAsync)
❌ N+1 query problems (missing Include statements)
❌ Not using AsNoTracking for read-only queries
❌ **Excessive comments** - Write self-documenting code with clear naming; avoid obvious comments

---

## Navigation Guide (⚠️ MANDATORY - Check This Table and Read Relevant Resources)

Use this table to determine which resource file you MUST read based on your task:

| If You Need To... | Then MUST Read... | When to Read It |
|-------------------|-------------------|-----------------|
| Understand architecture | [architecture-overview.md](resources/architecture-overview.md) | Before starting any new feature |
| Create controllers, endpoints, routes | [controllers-and-routing.md](resources/controllers-and-routing.md) | When creating or modifying API endpoints |
| Organize business logic, services | [services-and-repositories.md](resources/services-and-repositories.md) | When implementing service or repository methods |
| Validate input (DTOs, FluentValidation) | [validation-patterns.md](resources/validation-patterns.md) | When creating DTOs or validation logic |
| Handle errors, exceptions | [error-handling.md](resources/error-handling.md) | When implementing error handling |
| **Database queries, filtering, Include, performance** | **[entity-framework-patterns.md](resources/entity-framework-patterns.md)** | **⚡ CRITICAL: When writing ANY EF Core query** |
| Configure DI, services | [dependency-injection.md](resources/dependency-injection.md) | When registering services or configuring DI |
| Write tests | [testing-guide.md](resources/testing-guide.md) | When creating unit or integration tests |
| See complete examples | [complete-examples.md](resources/complete-examples.md) | When implementing full features |

**⚡ Most Common Mistake:** Writing database queries without reading entity-framework-patterns.md first, leading to poor performance (loading all data then filtering in memory).

---

## Resource Files

### [architecture-overview.md](resources/architecture-overview.md)
Layered architecture, request lifecycle, separation of concerns

### [controllers-and-routing.md](resources/controllers-and-routing.md)
API controllers, routing, HTTP methods, action results

### [services-and-repositories.md](resources/services-and-repositories.md)
Service patterns, DI, repository pattern, unit of work

### [validation-patterns.md](resources/validation-patterns.md)
FluentValidation, data annotations, custom validation

### [error-handling.md](resources/error-handling.md)
Global exception middleware, custom exceptions, logging

### [entity-framework-patterns.md](resources/entity-framework-patterns.md)
DbContext, migrations, queries, relationships, performance

### [dependency-injection.md](resources/dependency-injection.md)
Service registration, lifetimes, configuration

### [testing-guide.md](resources/testing-guide.md)
Unit tests, integration tests, mocking

### [complete-examples.md](resources/complete-examples.md)
Full examples, best practices

---

## Related Skills

- **angular-dev-guidelines**: Frontend Angular patterns that consume these APIs

---

**Skill Status**: Adapted for .NET Core 8 / ASP.NET Core
