# Validation Patterns in .NET Core 8

## Table of Contents
- [Overview](#overview)
- [Validation Approaches](#validation-approaches)
- [FluentValidation Setup](#fluentvalidation-setup)
- [Advanced FluentValidation Patterns](#advanced-fluentvalidation-patterns)
- [Validation in Different Layers](#validation-in-different-layers)
- [Error Response Patterns](#error-response-patterns)
- [Global Validation Error Handling](#global-validation-error-handling)
- [Testing Validators](#testing-validators)
- [Common Validation Patterns](#common-validation-patterns)
- [Best Practices](#best-practices)
- [See Also](#see-also)

---

## Overview

Input validation is critical for API security and data integrity. .NET Core provides multiple validation approaches.

## Validation Approaches

### 1. Data Annotations (Simple Cases)

```csharp
public class CreateProductDto
{
    [Required(ErrorMessage = "Product name is required")]
    [StringLength(200, MinimumLength = 3, ErrorMessage = "Name must be between 3 and 200 characters")]
    public string Name { get; set; } = string.Empty;

    [Required]
    [Range(0.01, double.MaxValue, ErrorMessage = "Price must be greater than 0")]
    public decimal Price { get; set; }

    [StringLength(50)]
    public string? Sku { get; set; }

    [Required]
    [EmailAddress(ErrorMessage = "Invalid email address")]
    public string SupplierEmail { get; set; } = string.Empty;

    [Url(ErrorMessage = "Invalid URL")]
    public string? WebsiteUrl { get; set; }

    [Phone(ErrorMessage = "Invalid phone number")]
    public string? Phone { get; set; }

    [RegularExpression(@"^[A-Z]{2}\d{6}$", ErrorMessage = "Invalid product code format")]
    public string? ProductCode { get; set; }
}
```

### 2. FluentValidation (Recommended for Complex Validation)

```csharp
using FluentValidation;

public class CreateProductDtoValidator : AbstractValidator<CreateProductDto>
{
    public CreateProductDtoValidator()
    {
        RuleFor(x => x.Name)
            .NotEmpty().WithMessage("Product name is required")
            .Length(3, 200).WithMessage("Name must be between 3 and 200 characters")
            .Must(BeValidProductName).WithMessage("Product name contains invalid characters");

        RuleFor(x => x.Price)
            .GreaterThan(0).WithMessage("Price must be greater than 0")
            .LessThanOrEqualTo(1000000).WithMessage("Price cannot exceed 1,000,000");

        RuleFor(x => x.Sku)
            .MaximumLength(50).WithMessage("SKU cannot exceed 50 characters")
            .Matches(@"^[A-Z0-9-]+$").WithMessage("SKU can only contain uppercase letters, numbers, and hyphens")
            .When(x => !string.IsNullOrEmpty(x.Sku));

        RuleFor(x => x.CategoryId)
            .GreaterThan(0).WithMessage("Category is required")
            .MustAsync(CategoryExists).WithMessage("Category does not exist");

        RuleFor(x => x.SupplierEmail)
            .NotEmpty().WithMessage("Supplier email is required")
            .EmailAddress().WithMessage("Invalid email format");

        RuleFor(x => x.Stock)
            .GreaterThanOrEqualTo(0).WithMessage("Stock cannot be negative");
    }

    private bool BeValidProductName(string name)
    {
        // Custom validation logic
        return !name.Contains("<") && !name.Contains(">");
    }

    private async Task<bool> CategoryExists(long categoryId, CancellationToken cancellationToken)
    {
        // Async validation against database
        using var scope = _serviceScopeFactory.CreateScope();
        var context = scope.ServiceProvider.GetRequiredService<ApplicationDbContext>();
        return await context.Categories.AnyAsync(c => c.Id == categoryId, cancellationToken);
    }
}
```

## FluentValidation Setup

### Registration in Program.cs

```csharp
using FluentValidation;
using FluentValidation.AspNetCore;

var builder = WebApplication.CreateBuilder(args);

// Register FluentValidation
builder.Services.AddFluentValidationAutoValidation();
builder.Services.AddFluentValidationClientsideAdapters();
builder.Services.AddValidatorsFromAssemblyContaining<CreateProductDtoValidator>();

// Or register individually
builder.Services.AddScoped<IValidator<CreateProductDto>, CreateProductDtoValidator>();
```

### Automatic Validation in Controllers

```csharp
[ApiController]
[Route("api/v{version:apiVersion}/[controller]")]
public class ProductController : ControllerBase
{
    [HttpPost]
    public async Task<IActionResult> Create([FromBody] CreateProductDto dto)
    {
        // ModelState automatically validated by FluentValidation
        if (!ModelState.IsValid)
        {
            return BadRequest(ModelState);
        }

        // Proceed with creation
        var result = await _productService.CreateAsync(dto);
        return CreatedAtAction(nameof(GetById), new { id = result.Id }, result);
    }
}
```

### Manual Validation

```csharp
public class ProductService(
    IValidator<CreateProductDto> validator,
    IProductRepository repository) : IProductService
{
    public async Task<ProductDto> CreateAsync(CreateProductDto dto)
    {
        // Manual validation
        var validationResult = await validator.ValidateAsync(dto);

        if (!validationResult.IsValid)
        {
            throw new ValidationException(validationResult.Errors);
        }

        // Proceed with business logic
        var product = new Product
        {
            Name = dto.Name,
            Price = dto.Price
        };

        await repository.AddAsync(product);
        return new ProductDto { Id = product.Id, Name = product.Name };
    }
}
```

## Advanced FluentValidation Patterns

### 1. Conditional Validation

```csharp
public class CreateOrderDtoValidator : AbstractValidator<CreateOrderDto>
{
    public CreateOrderDtoValidator()
    {
        // Validate shipping address only if not a digital product
        RuleFor(x => x.ShippingAddress)
            .NotEmpty()
            .When(x => x.RequiresShipping);

        // Different rules based on order type
        When(x => x.OrderType == OrderType.Wholesale, () =>
        {
            RuleFor(x => x.MinimumQuantity)
                .GreaterThanOrEqualTo(100)
                .WithMessage("Wholesale orders require minimum 100 units");
        }).Otherwise(() =>
        {
            RuleFor(x => x.MinimumQuantity)
                .Null()
                .WithMessage("Retail orders cannot have minimum quantity");
        });
    }
}
```

### 2. Collection Validation

```csharp
public class CreateOrderDtoValidator : AbstractValidator<CreateOrderDto>
{
    public CreateOrderDtoValidator()
    {
        RuleFor(x => x.Items)
            .NotEmpty().WithMessage("Order must have at least one item")
            .Must(items => items.Count <= 100).WithMessage("Order cannot exceed 100 items");

        RuleForEach(x => x.Items)
            .SetValidator(new OrderItemDtoValidator());
    }
}

public class OrderItemDtoValidator : AbstractValidator<OrderItemDto>
{
    public OrderItemDtoValidator()
    {
        RuleFor(x => x.ProductId)
            .GreaterThan(0).WithMessage("Product ID is required");

        RuleFor(x => x.Quantity)
            .GreaterThan(0).WithMessage("Quantity must be at least 1")
            .LessThanOrEqualTo(1000).WithMessage("Quantity cannot exceed 1000");

        RuleFor(x => x.Price)
            .GreaterThan(0).WithMessage("Price must be greater than 0");
    }
}
```

### 3. Cross-Property Validation

```csharp
public class UpdateProductPriceDtoValidator : AbstractValidator<UpdateProductPriceDto>
{
    public UpdateProductPriceDto Validator()
    {
        RuleFor(x => x.RegularPrice)
            .GreaterThan(0).WithMessage("Regular price must be greater than 0");

        RuleFor(x => x.SalePrice)
            .LessThan(x => x.RegularPrice)
            .WithMessage("Sale price must be less than regular price")
            .When(x => x.SalePrice.HasValue);

        RuleFor(x => x)
            .Must(x => x.SaleEndDate > x.SaleStartDate)
            .WithMessage("Sale end date must be after start date")
            .When(x => x.SaleStartDate.HasValue && x.SaleEndDate.HasValue);
    }
}
```

### 4. Custom Validators

```csharp
public class UniqueSku Validator : AbstractValidator<CreateProductDto>
{
    private readonly IProductRepository _repository;

    public UniqueSkuValidator(IProductRepository repository)
    {
        _repository = repository;

        RuleFor(x => x.Sku)
            .MustAsync(BeUniqueSku)
            .WithMessage("SKU already exists")
            .When(x => !string.IsNullOrEmpty(x.Sku));
    }

    private async Task<bool> BeUniqueSku(string sku, CancellationToken cancellationToken)
    {
        return !await _repository.ExistsBySkuAsync(sku);
    }
}
```

### 5. Custom Validation Extensions

```csharp
public static class CustomValidators
{
    public static IRuleBuilderOptions<T, string> IsValidProductCode<T>(
        this IRuleBuilder<T, string> ruleBuilder)
    {
        return ruleBuilder
            .NotEmpty()
            .Matches(@"^[A-Z]{2}\d{6}$")
            .WithMessage("Product code must be 2 uppercase letters followed by 6 digits");
    }

    public static IRuleBuilderOptions<T, DateTime> NotInPast<T>(
        this IRuleBuilder<T, DateTime> ruleBuilder)
    {
        return ruleBuilder
            .Must(date => date >= DateTime.UtcNow.Date)
            .WithMessage("Date cannot be in the past");
    }
}

// Usage
RuleFor(x => x.ProductCode).IsValidProductCode();
RuleFor(x => x.ExpiryDate).NotInPast();
```

## Validation in Different Layers

### 1. Controller Layer - Input Validation

```csharp
[ApiController]
[Route("api/v{version:apiVersion}/[controller]")]
public class ProductController : ControllerBase
{
    [HttpPost]
    public async Task<IActionResult> Create([FromBody] CreateProductDto dto)
    {
        // Automatic validation via FluentValidation or Data Annotations
        if (!ModelState.IsValid)
        {
            return BadRequest(new
            {
                Message = "Validation failed",
                Errors = ModelState.Values
                    .SelectMany(v => v.Errors)
                    .Select(e => e.ErrorMessage)
            });
        }

        var result = await _productService.CreateAsync(dto);
        return CreatedAtAction(nameof(GetById), new { id = result.Id }, result);
    }
}
```

### 2. Service Layer - Business Logic Validation

```csharp
public class ProductService : IProductService
{
    public async Task<ProductDto> CreateAsync(CreateProductDto dto)
    {
        // Business rule validation
        if (await _repository.ExistsByNameAsync(dto.Name))
        {
            throw new BusinessRuleViolationException("Product with this name already exists");
        }

        var category = await _categoryRepository.GetByIdAsync(dto.CategoryId);
        if (category == null)
        {
            throw new NotFoundException($"Category with ID {dto.CategoryId} not found");
        }

        if (category.IsActive == false)
        {
            throw new BusinessRuleViolationException("Cannot add product to inactive category");
        }

        // Create product
        var product = _mapper.Map<Product>(dto);
        await _repository.AddAsync(product);

        return _mapper.Map<ProductDto>(product);
    }
}
```

### 3. Domain Layer - Entity Validation

```csharp
public class Product : AuditedEntity
{
    private decimal _price;
    private int _stock;

    public string Name { get; set; } = string.Empty;

    public decimal Price
    {
        get => _price;
        set
        {
            if (value <= 0)
                throw new ArgumentException("Price must be greater than 0");
            _price = value;
        }
    }

    public int Stock
    {
        get => _stock;
        set
        {
            if (value < 0)
                throw new ArgumentException("Stock cannot be negative");
            _stock = value;
        }
    }

    public void UpdatePrice(decimal newPrice)
    {
        if (newPrice <= 0)
            throw new ArgumentException("Price must be greater than 0");

        if (newPrice > _price * 2)
            throw new BusinessRuleViolationException("Price cannot be increased by more than 100%");

        _price = newPrice;
    }
}
```

## Error Response Patterns

### Standard Error Response DTO

```csharp
public class ApiResponse<T>
{
    public bool Success { get; set; }
    public string Message { get; set; } = string.Empty;
    public T? Data { get; set; }
    public List<string>? Errors { get; set; }

    public static ApiResponse<T> SuccessResult(T data, string message = "Operation successful")
    {
        return new ApiResponse<T>
        {
            Success = true,
            Message = message,
            Data = data
        };
    }

    public static ApiResponse<T> ErrorResult(string message, List<string>? errors = null)
    {
        return new ApiResponse<T>
        {
            Success = false,
            Message = message,
            Errors = errors
        };
    }
}
```

### Validation Error Response

```csharp
public class ValidationErrorResponse
{
    public string Message { get; set; } = "Validation failed";
    public Dictionary<string, List<string>> Errors { get; set; } = new();

    public static ValidationErrorResponse FromModelState(ModelStateDictionary modelState)
    {
        var response = new ValidationErrorResponse();

        foreach (var (key, value) in modelState)
        {
            var errors = value.Errors.Select(e => e.ErrorMessage).ToList();
            if (errors.Any())
            {
                response.Errors[key] = errors;
            }
        }

        return response;
    }

    public static ValidationErrorResponse FromFluentValidation(ValidationException exception)
    {
        var response = new ValidationErrorResponse();

        foreach (var error in exception.Errors)
        {
            if (!response.Errors.ContainsKey(error.PropertyName))
            {
                response.Errors[error.PropertyName] = new List<string>();
            }

            response.Errors[error.PropertyName].Add(error.ErrorMessage);
        }

        return response;
    }
}
```

## Global Validation Error Handling

### Middleware for Validation Exceptions

```csharp
public class ValidationExceptionMiddleware
{
    private readonly RequestDelegate _next;

    public ValidationExceptionMiddleware(RequestDelegate next)
    {
        _next = next;
    }

    public async Task InvokeAsync(HttpContext context)
    {
        try
        {
            await _next(context);
        }
        catch (ValidationException validationException)
        {
            context.Response.StatusCode = StatusCodes.Status400BadRequest;
            context.Response.ContentType = "application/json";

            var response = ValidationErrorResponse.FromFluentValidation(validationException);
            await context.Response.WriteAsJsonAsync(response);
        }
    }
}

// Register in Program.cs
app.UseMiddleware<ValidationExceptionMiddleware>();
```

## Testing Validators

### Unit Testing FluentValidation

```csharp
public class CreateProductDtoValidatorTests
{
    private readonly CreateProductDtoValidator _validator;

    public CreateProductDtoValidatorTests()
    {
        _validator = new CreateProductDtoValidator();
    }

    [Fact]
    public void Should_Have_Error_When_Name_Is_Empty()
    {
        var dto = new CreateProductDto { Name = "" };

        var result = _validator.TestValidate(dto);

        result.ShouldHaveValidationErrorFor(x => x.Name)
            .WithErrorMessage("Product name is required");
    }

    [Fact]
    public void Should_Have_Error_When_Price_Is_Zero()
    {
        var dto = new CreateProductDto { Price = 0 };

        var result = _validator.TestValidate(dto);

        result.ShouldHaveValidationErrorFor(x => x.Price);
    }

    [Fact]
    public void Should_Not_Have_Error_When_Valid()
    {
        var dto = new CreateProductDto
        {
            Name = "Test Product",
            Price = 99.99m,
            CategoryId = 1
        };

        var result = _validator.TestValidate(dto);

        result.ShouldNotHaveAnyValidationErrors();
    }
}
```

## Common Validation Patterns

### Email Validation

```csharp
RuleFor(x => x.Email)
    .NotEmpty()
    .EmailAddress()
    .MaximumLength(255);
```

### Phone Number Validation

```csharp
RuleFor(x => x.Phone)
    .Matches(@"^\+?[1-9]\d{1,14}$")
    .WithMessage("Phone number must be in E.164 format")
    .When(x => !string.IsNullOrEmpty(x.Phone));
```

### Date Range Validation

```csharp
RuleFor(x => x.StartDate)
    .LessThan(x => x.EndDate)
    .WithMessage("Start date must be before end date");

RuleFor(x => x.EndDate)
    .GreaterThan(DateTime.UtcNow)
    .WithMessage("End date must be in the future");
```

### Enum Validation

```csharp
RuleFor(x => x.Status)
    .IsInEnum()
    .WithMessage("Invalid status value");
```

## Best Practices

✅ **Validate at multiple layers** - Input validation in DTOs, business rules in services, invariants in entities

✅ **Use FluentValidation for complex scenarios** - More flexible and testable than Data Annotations

✅ **Async validation for database checks** - Use `MustAsync` for database lookups

✅ **Clear error messages** - User-friendly messages that explain what's wrong

✅ **Centralized error handling** - Use middleware for consistent error responses

✅ **Test validators** - Write unit tests for all validation rules

❌ **Don't skip validation** - Always validate input, never trust client-side validation alone

❌ **Don't mix concerns** - Keep validation separate from business logic

❌ **Don't over-validate** - Balance security with user experience

## See Also

- [error-handling.md](error-handling.md) - Global exception handling
- [controllers-and-routing.md](controllers-and-routing.md) - Controller patterns
- [testing-guide.md](testing-guide.md) - Testing validators
