# Services and Repositories

Complete guide to service layer and repository pattern in .NET Core 8 applications.

## Table of Contents
- [Service Layer Pattern](#service-layer-pattern)
- [Repository Pattern](#repository-pattern)
- [Unit of Work Pattern](#unit-of-work-pattern)
- [Dependency Injection Registration](#dependency-injection-registration)
- [Service Lifetimes](#service-lifetimes)
- [Best Practices](#best-practices)
- [Summary](#summary)

---

## Service Layer Pattern

### Why Services?

**Benefits:**
- Separation of concerns
- Business logic centralization
- Reusable across controllers
- Easier to test
- Independent of HTTP context

### Service Interface

```csharp
// Services/Interfaces/IUserService.cs
namespace MyApi.Services.Interfaces;

public interface IUserService
{
    Task<IEnumerable<UserResponseDto>> GetAllUsers();
    Task<UserResponseDto?> GetUserById(int id);
    Task<UserResponseDto> CreateUser(CreateUserDto dto);
    Task<bool> UpdateUser(int id, UpdateUserDto dto);
    Task<bool> DeleteUser(int id);
    Task<bool> UserExists(int id);
}
```

### Service Implementation

```csharp
public class UserService : IUserService
{
    private readonly IUserRepository _userRepository;
    private readonly IMapper _mapper;
    private readonly ILogger<UserService> _logger;

    public UserService(IUserRepository userRepository, IMapper mapper, ILogger<UserService> logger)
    {
        _userRepository = userRepository;
        _mapper = mapper;
        _logger = logger;
    }

    public async Task<UserResponseDto> CreateUser(CreateUserDto dto)
    {
        if (await _userRepository.EmailExistsAsync(dto.Email))
            throw new InvalidOperationException("Email already exists");

        var user = _mapper.Map<User>(dto);
        user.CreatedAt = DateTime.UtcNow;

        var created = await _userRepository.AddAsync(user);
        await _userRepository.SaveChangesAsync();

        _logger.LogInformation("User created with ID: {UserId}", created.Id);
        return _mapper.Map<UserResponseDto>(created);
    }

    public async Task<bool> DeleteUser(int id)
    {
        var user = await _userRepository.GetByIdAsync(id);
        if (user == null) return false;

        _userRepository.Delete(user);
        await _userRepository.SaveChangesAsync();
        return true;
    }
}
```

---

## Repository Pattern

### Why Repositories?

**Benefits:**
- Abstraction over data access
- Centralized query logic
- Easier to mock for testing
- Can switch data sources
- Cleaner service layer

### ⚡ CRITICAL: Repository Returns IQueryable, Service Applies Pagination

**Architectural Rule:**
- **Repositories** return `IQueryable<T>` for filtering methods (NO pagination)
- **Services** apply pagination using `ToPagedResultAsync()` extension method
- This ensures flexibility, reusability, and proper separation of concerns

### Repository Interface

```csharp
// Repositories/Interfaces/IRepository.cs
namespace MyApi.Repositories.Interfaces;

public interface IRepository<T> where T : class
{
    // ✅ Return IQueryable for flexibility (no pagination)
    IQueryable<T> GetAll();

    Task<T?> GetById(int id);
    Task<T> Add(T entity);
    void Update(T entity);
    void Delete(T entity);
    Task<bool> Exists(int id);
    Task SaveChangesAsync();
}

// Repositories/Interfaces/IUserRepository.cs
public interface IUserRepository : IRepository<User>
{
    Task<User?> GetByEmail(string email);

    // ✅ CORRECT - Returns IQueryable for service to paginate
    IQueryable<User> GetActiveUsers(UserFilter filter);

    Task<bool> EmailExists(string email);

    // ❌ WRONG - Don't return PagedResult from repository
    // Task<PagedResult<User>> GetActiveUsers(UserFilter filter);
}
```

### Repository Implementation

```csharp
public class Repository<T> : IRepository<T> where T : class
{
    protected readonly ApplicationDbContext _context;
    protected readonly DbSet<T> _dbSet;

    public Repository(ApplicationDbContext context)
    {
        _context = context;
        _dbSet = context.Set<T>();
    }

    public virtual IQueryable<T> GetAll() => _dbSet.AsNoTracking();
    public virtual async Task<T?> GetById(int id) => await _dbSet.FindAsync(id);
    public virtual async Task<T> Add(T entity)
    {
        await _dbSet.AddAsync(entity);
        return entity;
    }
    public virtual void Update(T entity) => _dbSet.Update(entity);
    public virtual void Delete(T entity) => _dbSet.Remove(entity);
    public virtual async Task SaveChangesAsync() => await _context.SaveChangesAsync();
}

public class UserRepository : Repository<User>, IUserRepository
{
    public UserRepository(ApplicationDbContext context) : base(context) { }

    public IQueryable<User> GetActiveUsers(UserFilter filter)
    {
        var query = _dbSet.AsNoTracking().Where(u => u.IsActive);

        if (!string.IsNullOrEmpty(filter.SearchTerm))
            query = query.Where(u => u.Name.Contains(filter.SearchTerm));

        return query; // NO pagination/sorting
    }

    public async Task<bool> EmailExists(string email) =>
        await _dbSet.AnyAsync(u => u.Email == email);
}
```

### Service with Pagination

```csharp
public async Task<PagedResult<UserDto>> GetActiveUsersAsync(UserFilter filter)
{
    var query = _userRepository.GetActiveUsers(filter);
    query = query.OrderBy(u => u.Name); // Service adds sorting
    return await query.ToPagedResultAsync(filter.PageNumber, filter.PageSize);
}
```

---

## Unit of Work Pattern

### When to Use

Use Unit of Work when you need to:
- Coordinate multiple repository operations
- Ensure transactional consistency
- Manage DbContext lifecycle explicitly

### Unit of Work Interface

```csharp
// Repositories/Interfaces/IUnitOfWork.cs
namespace MyApi.Repositories.Interfaces;

public interface IUnitOfWork : IDisposable
{
    IUserRepository Users { get; }
    IOrderRepository Orders { get; }
    IProductRepository Products { get; }

    Task<int> SaveChangesAsync();
    Task BeginTransactionAsync();
    Task CommitTransactionAsync();
    Task RollbackTransactionAsync();
}
```

### Unit of Work Implementation

```csharp
// Repositories/Implementations/UnitOfWork.cs
using Microsoft.EntityFrameworkCore.Storage;

namespace MyApi.Repositories.Implementations;

public class UnitOfWork : IUnitOfWork
{
    private readonly ApplicationDbContext _context;
    private IDbContextTransaction? _transaction;

    public IUserRepository Users { get; }
    public IOrderRepository Orders { get; }
    public IProductRepository Products { get; }

    public UnitOfWork(
        ApplicationDbContext context,
        IUserRepository userRepository,
        IOrderRepository orderRepository,
        IProductRepository productRepository)
    {
        _context = context;
        Users = userRepository;
        Orders = orderRepository;
        Products = productRepository;
    }

    public async Task<int> SaveChangesAsync()
    {
        return await _context.SaveChangesAsync();
    }

    public async Task BeginTransactionAsync()
    {
        _transaction = await _context.Database.BeginTransactionAsync();
    }

    public async Task CommitTransactionAsync()
    {
        if (_transaction != null)
        {
            await _transaction.CommitAsync();
            await _transaction.DisposeAsync();
            _transaction = null;
        }
    }

    public async Task RollbackTransactionAsync()
    {
        if (_transaction != null)
        {
            await _transaction.RollbackAsync();
            await _transaction.DisposeAsync();
            _transaction = null;
        }
    }

    public void Dispose()
    {
        _transaction?.Dispose();
        _context.Dispose();
    }
}
```

### Using Unit of Work

```csharp
public async Task<OrderResponseDto> CreateOrderAsync(CreateOrderDto dto)
{
    await _unitOfWork.BeginTransactionAsync();

    try
    {
        var order = new Order { UserId = dto.UserId, TotalAmount = 0 };
        await _unitOfWork.Orders.AddAsync(order);
        await _unitOfWork.SaveChangesAsync();

        foreach (var item in dto.Items)
        {
            var product = await _unitOfWork.Products.GetByIdAsync(item.ProductId);
            if (product == null || product.Stock < item.Quantity)
                throw new InvalidOperationException("Insufficient stock");

            product.Stock -= item.Quantity;
            _unitOfWork.Products.Update(product);
        }

        await _unitOfWork.SaveChangesAsync();
        await _unitOfWork.CommitTransactionAsync();
        return _mapper.Map<OrderResponseDto>(order);
    }
    catch
    {
        await _unitOfWork.RollbackTransactionAsync();
        throw;
    }
}
```

---

## Dependency Injection Registration

### Program.cs Registration

```csharp
// Program.cs
var builder = WebApplication.CreateBuilder(args);

// DbContext
builder.Services.AddDbContext<ApplicationDbContext>(options =>
    options.UseSqlServer(builder.Configuration.GetConnectionString("DefaultConnection")));

// Repositories
builder.Services.AddScoped<IUserRepository, UserRepository>();
builder.Services.AddScoped<IOrderRepository, OrderRepository>();
builder.Services.AddScoped<IProductRepository, ProductRepository>();

// Services
builder.Services.AddScoped<IUserService, UserService>();
builder.Services.AddScoped<IOrderService, OrderService>();
builder.Services.AddScoped<IProductService, ProductService>();

// Unit of Work (optional)
builder.Services.AddScoped<IUnitOfWork, UnitOfWork>();

// AutoMapper
builder.Services.AddAutoMapper(typeof(Program));

var app = builder.Build();
```

---

## Service Lifetimes

### Transient
Created each time they're requested
```csharp
builder.Services.AddTransient<IEmailService, EmailService>();
```

### Scoped (RECOMMENDED for most services)
Created once per request
```csharp
builder.Services.AddScoped<IUserService, UserService>();
builder.Services.AddScoped<IUserRepository, UserRepository>();
```

### Singleton
Created once for application lifetime
```csharp
builder.Services.AddSingleton<ICacheService, CacheService>();
```

---

## Best Practices

### 1. Services Handle Business Logic

```csharp
// ✅ Services contain validation and business rules
public async Task<UserResponseDto> CreateUserAsync(CreateUserDto dto)
{
    if (await _userRepository.EmailExistsAsync(dto.Email))
        throw new InvalidOperationException("Email already exists");

    var user = _mapper.Map<User>(dto);
    user.CreatedAt = DateTime.UtcNow;
    await _userRepository.AddAsync(user);
    await _userRepository.SaveChangesAsync();
    return _mapper.Map<UserResponseDto>(user);
}
```

### 2. Repositories Return IQueryable

```csharp
// ✅ Repository returns IQueryable (no pagination)
public IQueryable<User> GetActiveUsers(UserFilter filter)
{
    var query = _dbSet.AsNoTracking().Where(u => u.IsActive);
    if (!string.IsNullOrEmpty(filter.SearchTerm))
        query = query.Where(u => u.Name.Contains(filter.SearchTerm));
    return query;
}

// ✅ Service applies pagination
public async Task<PagedResult<UserDto>> GetActiveUsersAsync(UserFilter filter)
{
    var query = _userRepository.GetActiveUsers(filter).OrderBy(u => u.Name);
    return await query.ToPagedResultAsync(filter.PageNumber, filter.PageSize);
}
```

### 3. Use DTOs in Services

```csharp
// ✅ Return DTOs, not entities
public async Task<UserResponseDto> GetUserByIdAsync(int id)
{
    var user = await _userRepository.GetByIdAsync(id);
    return _mapper.Map<UserResponseDto>(user);
}
```

---

## Summary

**Service Layer:**
- Contains business logic
- Uses DTOs
- Delegates data access to repositories
- Handles validation
- Logging and error handling
- ⚡ **Applies pagination using `ToPagedResultAsync()`**
- ⚡ **Applies sorting to queries**

**Repository Layer:**
- Abstracts data access
- Encapsulates queries
- No business logic
- ⚡ **Returns `IQueryable<T>` for filtering methods (NO pagination)**
- ⚡ **Applies filters only (NO sorting, NO Skip/Take)**
- Returns entities (not DTOs)

**Unit of Work:**
- Coordinates multiple repositories
- Manages transactions
- Ensures data consistency

**See Also:**
- [controllers-and-routing.md](controllers-and-routing.md) - Controller patterns
- [entity-framework-patterns.md](entity-framework-patterns.md) - EF Core patterns
- [dependency-injection.md](dependency-injection.md) - DI configuration
- [complete-examples.md](complete-examples.md) - Full examples
