# Error Handling in .NET Core 8

## Table of Contents
- [Overview](#overview)
- [Global Exception Middleware](#global-exception-middleware)
- [Custom Exception Types](#custom-exception-types)
- [Error Response Format](#error-response-format)
- [HTTP Status Code Guidelines](#http-status-code-guidelines)
- [Logging Best Practices](#logging-best-practices)
- [Validation Errors](#validation-errors)
- [Problem Details (RFC 7807)](#problem-details-rfc-7807)
- [Best Practices](#best-practices)
- [Common Patterns](#common-patterns)
- [See Also](#see-also)

---

## Overview

Proper error handling ensures:
- Consistent error responses across all endpoints
- Security (no sensitive data leaked)
- Clear debugging information (in logs, not responses)
- Good user experience with meaningful error messages

**Key Principle:** Handle exceptions globally, not in individual controllers.

---

## Global Exception Middleware

### Exception Handling Middleware

```csharp
// Middleware/ExceptionHandlingMiddleware.cs
public class ExceptionHandlingMiddleware
{
    private readonly RequestDelegate _next;
    private readonly ILogger<ExceptionHandlingMiddleware> _logger;
    private readonly IHostEnvironment _environment;

    public ExceptionHandlingMiddleware(
        RequestDelegate next,
        ILogger<ExceptionHandlingMiddleware> logger,
        IHostEnvironment environment)
    {
        _next = next;
        _logger = logger;
        _environment = environment;
    }

    public async Task InvokeAsync(HttpContext context)
    {
        try
        {
            await _next(context);
        }
        catch (Exception ex)
        {
            await HandleExceptionAsync(context, ex);
        }
    }

    private async Task HandleExceptionAsync(HttpContext context, Exception exception)
    {
        _logger.LogError(exception, "An unhandled exception occurred: {Message}", exception.Message);

        var (statusCode, message) = exception switch
        {
            NotFoundException => (StatusCodes.Status404NotFound, exception.Message),
            ValidationException => (StatusCodes.Status400BadRequest, exception.Message),
            UnauthorizedException => (StatusCodes.Status401Unauthorized, "Unauthorized access"),
            ForbiddenException => (StatusCodes.Status403Forbidden, "Access forbidden"),
            ConflictException => (StatusCodes.Status409Conflict, exception.Message),
            _ => (StatusCodes.Status500InternalServerError,
                  _environment.IsDevelopment()
                      ? exception.Message
                      : "An internal server error occurred")
        };

        context.Response.ContentType = "application/json";
        context.Response.StatusCode = statusCode;

        var response = new ErrorResponse
        {
            StatusCode = statusCode,
            Message = message,
            Details = _environment.IsDevelopment() ? exception.StackTrace : null
        };

        var json = JsonSerializer.Serialize(response, new JsonSerializerOptions
        {
            PropertyNamingPolicy = JsonNamingPolicy.CamelCase
        });

        await context.Response.WriteAsync(json);
    }
}
```

### Register Middleware

```csharp
// Program.cs
var app = builder.Build();

// Exception handling must be FIRST in pipeline
app.UseMiddleware<ExceptionHandlingMiddleware>();

app.UseAuthentication();
app.UseAuthorization();
app.MapControllers();

app.Run();
```

---

## Custom Exception Types

### Base Business Exception

```csharp
// Core/Exceptions/BusinessException.cs
public abstract class BusinessException : Exception
{
    public int StatusCode { get; }

    protected BusinessException(string message, int statusCode) : base(message)
    {
        StatusCode = statusCode;
    }
}
```

### Specific Exception Types

```csharp
// Core/Exceptions/NotFoundException.cs
public class NotFoundException : BusinessException
{
    public NotFoundException(string entityName, object key)
        : base($"{entityName} with id '{key}' was not found", StatusCodes.Status404NotFound)
    {
    }
}

// Core/Exceptions/ValidationException.cs
public class ValidationException : BusinessException
{
    public IDictionary<string, string[]> Errors { get; }

    public ValidationException(IDictionary<string, string[]> errors)
        : base("One or more validation errors occurred", StatusCodes.Status400BadRequest)
    {
        Errors = errors;
    }
}

// Core/Exceptions/ConflictException.cs
public class ConflictException : BusinessException
{
    public ConflictException(string message)
        : base(message, StatusCodes.Status409Conflict)
    {
    }
}

// Core/Exceptions/UnauthorizedException.cs
public class UnauthorizedException : BusinessException
{
    public UnauthorizedException(string message = "Unauthorized access")
        : base(message, StatusCodes.Status401Unauthorized)
    {
    }
}

// Core/Exceptions/ForbiddenException.cs
public class ForbiddenException : BusinessException
{
    public ForbiddenException(string message = "Access forbidden")
        : base(message, StatusCodes.Status403Forbidden)
    {
    }
}
```

---

## Error Response Format

### Standard Error Response Model

```csharp
// API/Models/ErrorResponse.cs
public class ErrorResponse
{
    public int StatusCode { get; set; }
    public string Message { get; set; } = string.Empty;
    public string? Details { get; set; }
    public DateTime Timestamp { get; set; } = DateTime.UtcNow;
    public IDictionary<string, string[]>? ValidationErrors { get; set; }
}
```

### Example Responses

**404 Not Found:**
```json
{
  "statusCode": 404,
  "message": "Book with id '123' was not found",
  "timestamp": "2025-12-03T10:30:00Z"
}
```

**400 Validation Error:**
```json
{
  "statusCode": 400,
  "message": "One or more validation errors occurred",
  "timestamp": "2025-12-03T10:30:00Z",
  "validationErrors": {
    "name": ["Name is required", "Name must be between 3 and 100 characters"],
    "email": ["Email is not valid"]
  }
}
```

**500 Internal Error (Production):**
```json
{
  "statusCode": 500,
  "message": "An internal server error occurred",
  "timestamp": "2025-12-03T10:30:00Z"
}
```

**500 Internal Error (Development):**
```json
{
  "statusCode": 500,
  "message": "Object reference not set to an instance of an object",
  "details": "   at MyService.GetUser(Int32 id)...",
  "timestamp": "2025-12-03T10:30:00Z"
}
```

---

## HTTP Status Code Guidelines

| Code | Scenario | Exception Type |
|------|----------|----------------|
| 200 | Success | - |
| 201 | Created | - |
| 204 | No Content | - |
| 400 | Bad Request | ValidationException |
| 401 | Unauthorized | UnauthorizedException |
| 403 | Forbidden | ForbiddenException |
| 404 | Not Found | NotFoundException |
| 409 | Conflict | ConflictException |
| 500 | Server Error | Exception (unhandled) |

---

## Logging Best Practices

### Structured Logging

```csharp
// ✅ GOOD - Structured logging
_logger.LogError(exception,
    "Failed to create book for user {UserId} in unit {UnitId}",
    userId, unitId);

// ❌ BAD - String interpolation loses structure
_logger.LogError($"Failed to create book for user {userId}");
```

### Log Levels

```csharp
// Error - Exceptions and failures
_logger.LogError(exception, "Database connection failed");

// Warning - Unexpected but handled situations
_logger.LogWarning("Book {BookId} already exists, skipping creation", bookId);

// Information - Important business events
_logger.LogInformation("User {UserId} created book {BookId}", userId, bookId);

// Debug - Diagnostic information (development only)
_logger.LogDebug("Executing query: {Query}", query);
```

### What NOT to Log

❌ Sensitive data (passwords, tokens, personal info)
❌ Full request/response bodies (unless sanitized)
❌ Credit card numbers, social security numbers
❌ Full stack traces in production (use message only)

---

## Validation Errors

### FluentValidation Exception Handling

```csharp
// In middleware
catch (FluentValidation.ValidationException validationEx)
{
    var errors = validationEx.Errors
        .GroupBy(e => e.PropertyName)
        .ToDictionary(
            g => g.Key,
            g => g.Select(e => e.ErrorMessage).ToArray()
        );

    var response = new ErrorResponse
    {
        StatusCode = StatusCodes.Status400BadRequest,
        Message = "Validation failed",
        ValidationErrors = errors
    };

    context.Response.StatusCode = StatusCodes.Status400BadRequest;
    await context.Response.WriteAsJsonAsync(response);
}
```

---

## Problem Details (RFC 7807)

### Using Built-in Problem Details

```csharp
// Program.cs
builder.Services.AddProblemDetails();

// Middleware automatically returns Problem Details format
app.UseExceptionHandler();
app.UseStatusCodePages();
```

### Custom Problem Details

```csharp
public class CustomProblemDetailsFactory : ProblemDetailsFactory
{
    public override ProblemDetails CreateProblemDetails(
        HttpContext httpContext,
        int? statusCode = null,
        string? title = null,
        string? type = null,
        string? detail = null,
        string? instance = null)
    {
        var problemDetails = new ProblemDetails
        {
            Status = statusCode ?? 500,
            Title = title ?? "An error occurred",
            Type = type ?? "https://tools.ietf.org/html/rfc7231#section-6.6.1",
            Detail = detail,
            Instance = instance ?? httpContext.Request.Path
        };

        // Add custom properties
        problemDetails.Extensions["traceId"] = httpContext.TraceIdentifier;
        problemDetails.Extensions["timestamp"] = DateTime.UtcNow;

        return problemDetails;
    }
}

// Register
builder.Services.AddSingleton<ProblemDetailsFactory, CustomProblemDetailsFactory>();
```

---

## Best Practices

### ✅ DO

1. **Use global exception middleware** - Don't catch in controllers
2. **Log exceptions with context** - Include user ID, entity IDs, etc.
3. **Return consistent error format** - Same structure across all endpoints
4. **Hide sensitive errors in production** - Generic messages, detailed logs
5. **Use custom exception types** - Makes error handling easier
6. **Include correlation IDs** - For distributed tracing
7. **Validate early** - Catch validation errors before business logic

### ❌ DON'T

1. **Don't expose stack traces in production** - Security risk
2. **Don't log sensitive data** - Passwords, tokens, personal info
3. **Don't catch and ignore exceptions** - Always log or rethrow
4. **Don't use exceptions for flow control** - Use return values
5. **Don't return different error formats** - Inconsistent API
6. **Don't handle business exceptions in controllers** - Use middleware

---

## Common Patterns

### Controller Exception Handling - Minimal Catch Blocks

**✅ RECOMMENDED PATTERN:** Only catch specific business exceptions (like `InvalidOperationException`) in controllers. Let global middleware handle all other exceptions.

```csharp
// ✅ GOOD - Minimal controller exception handling
[HttpPost]
public async Task<ActionResult<ApiResponse<bool>>> Create([FromBody] CreateTemplateDto createDto)
{
    try
    {
        var result = await _templateService.Create(createDto);
        return Ok(ApiResponse<bool>.SuccessResult(result, "Template created successfully."));
    }
    catch (InvalidOperationException ex)
    {
        return BadRequest(ApiResponse<bool>.ErrorResult(ex.Message));
    }
    // No general Exception catch - let middleware handle it
}

// ✅ GOOD - No try-catch for GET operations
[HttpGet]
public async Task<ActionResult<ApiResponse<List<TemplateDto>>>> GetAll()
{
    var templates = await _templateService.GetAll();
    return Ok(ApiResponse<List<TemplateDto>>.SuccessResult(templates, "Templates retrieved successfully."));
}

// ❌ BAD - Don't catch general exceptions in controllers
[HttpPost]
public async Task<ActionResult<ApiResponse<bool>>> Create([FromBody] CreateTemplateDto createDto)
{
    try
    {
        var result = await _templateService.Create(createDto);
        return Ok(ApiResponse<bool>.SuccessResult(result, "Template created successfully."));
    }
    catch (InvalidOperationException ex)
    {
        return BadRequest(ApiResponse<bool>.ErrorResult(ex.Message));
    }
    catch (Exception ex) // ❌ Don't do this - middleware should handle
    {
        return StatusCode(500, ApiResponse<bool>.ErrorResult($"An error occurred: {ex.Message}"));
    }
}
```

**Why?**
- Global middleware provides consistent error handling across all endpoints
- Reduces code duplication
- Easier to modify error handling behavior globally
- Better logging and monitoring at the middleware level
- Only handle business-specific exceptions in controllers

### Service Layer Exception Throwing

```csharp
// BookService.cs
public async Task<BookDto> GetByIdAsync(long id)
{
    var book = await _bookRepository.GetByIdAsync(id);

    if (book == null)
        throw new NotFoundException(nameof(Book), id);

    return _mapper.Map<BookDto>(book);
}
```

### Repository Method with Validation

```csharp
// ApproverService.cs
public async Task<ApproverDto> CreateApproverAsync(CreateApproverDto dto)
{
    // Check for conflicts
    var exists = await _approverRepository.ExistsAsync(dto.EmployeeId);
    if (exists)
        throw new ConflictException($"Approver with employee ID {dto.EmployeeId} already exists");

    // Business logic
    var approver = _mapper.Map<Approver>(dto);
    await _approverRepository.AddAsync(approver);

    return _mapper.Map<ApproverDto>(approver);
}
```

### Authorization Check

```csharp
// BookService.cs
public async Task DeleteBookAsync(long bookId, long currentUserId)
{
    var book = await _bookRepository.GetByIdAsync(bookId);

    if (book == null)
        throw new NotFoundException(nameof(Book), bookId);

    // Authorization check
    if (book.CreatedBy != currentUserId)
        throw new ForbiddenException("You don't have permission to delete this book");

    await _bookRepository.DeleteAsync(bookId);
}
```

### Try-Catch for External Services

```csharp
// HRService.cs
public async Task<List<EmployeeDto>> GetEmployeesByIdsAsync(List<long> ids)
{
    try
    {
        var response = await _httpClient.GetAsync($"api/employees?ids={string.Join(",", ids)}");
        response.EnsureSuccessStatusCode();

        return await response.Content.ReadFromJsonAsync<List<EmployeeDto>>()
            ?? new List<EmployeeDto>();
    }
    catch (HttpRequestException ex)
    {
        _logger.LogError(ex, "Failed to fetch employees from HR service");
        throw new InvalidOperationException("Unable to connect to HR service", ex);
    }
}
```

---

## See Also

- [controllers-and-routing.md](controllers-and-routing.md) - Controller patterns
- [services-and-repositories.md](services-and-repositories.md) - Service implementation
- [validation-patterns.md](validation-patterns.md) - Input validation
- [complete-examples.md](complete-examples.md) - Full feature examples
