# Entity Framework Core Patterns

## Table of Contents
- [DbContext Setup](#dbcontext-setup)
- [Entity Configurations](#entity-configurations)
- [Base Entity Pattern](#base-entity-pattern)
- [Query Patterns](#query-patterns)
- [Performance Patterns](#performance-patterns)
- [Soft Delete Pattern](#soft-delete-pattern)
- [Transaction Handling](#transaction-handling)
- [Common Pitfalls](#common-pitfalls)
- [Advanced Patterns](#advanced-patterns)
- [See Also](#see-also)

---

## DbContext Setup

### Standard DbContext Structure

```csharp
public class ApplicationDbContext : DbContext
{
    public ApplicationDbContext(DbContextOptions<ApplicationDbContext> options)
        : base(options) { }

    // DbSets for your entities
    public DbSet<Product> Products { get; set; } = null!;
    public DbSet<Order> Orders { get; set; } = null!;
    public DbSet<Customer> Customers { get; set; } = null!;

    protected override void OnModelCreating(ModelBuilder modelBuilder)
    {
        base.OnModelCreating(modelBuilder);

        // Apply global query filters
        ConfigureGlobalQueryFilters(modelBuilder);

        // Apply entity configurations from assembly
        modelBuilder.ApplyConfigurationsFromAssembly(typeof(ApplicationDbContext).Assembly);
    }

    // Soft delete pattern - automatically exclude deleted records
    private void ConfigureGlobalQueryFilters(ModelBuilder modelBuilder)
    {
        foreach (var entityType in modelBuilder.Model.GetEntityTypes())
        {
            if (typeof(ISoftDeletable).IsAssignableFrom(entityType.ClrType))
            {
                var parameter = Expression.Parameter(entityType.ClrType, "e");
                var property = Expression.Property(parameter, nameof(ISoftDeletable.IsDeleted));
                var filter = Expression.Lambda(Expression.Not(property), parameter);

                modelBuilder.Entity(entityType.ClrType).HasQueryFilter(filter);
            }
        }
    }

    // Audit trail automation - automatically set timestamps
    public override async Task<int> SaveChangesAsync(CancellationToken cancellationToken = default)
    {
        var currentUserId = GetCurrentUserId(); // From HttpContext or JWT

        foreach (var entry in ChangeTracker.Entries<AuditedEntity>())
        {
            if (entry.State == EntityState.Added)
            {
                entry.Entity.CreatedAt = DateTime.UtcNow;
                entry.Entity.CreatedBy = currentUserId;
            }
            else if (entry.State == EntityState.Modified)
            {
                entry.Entity.UpdatedAt = DateTime.UtcNow;
            }
        }

        return await base.SaveChangesAsync(cancellationToken);
    }

    private long? GetCurrentUserId()
    {
        // Get from HttpContextAccessor or current user context
        return _httpContextAccessor?.HttpContext?.User?.FindFirst("userId")?.Value != null
            ? long.Parse(_httpContextAccessor.HttpContext.User.FindFirst("userId")!.Value)
            : null;
    }
}
```

## Entity Configurations

```csharp
public class ProductConfiguration : IEntityTypeConfiguration<Product>
{
    public void Configure(EntityTypeBuilder<Product> builder)
    {
        builder.ToTable("Products");
        builder.HasKey(p => p.Id);

        builder.Property(p => p.Name).IsRequired().HasMaxLength(200);
        builder.Property(p => p.Price).HasColumnType("decimal(18,2)");
        builder.Property(p => p.Status).HasConversion<string>().HasMaxLength(50);

        builder.HasIndex(p => p.Sku).IsUnique();

        builder.HasOne(p => p.Category)
            .WithMany(c => c.Products)
            .HasForeignKey(p => p.CategoryId)
            .OnDelete(DeleteBehavior.Restrict);
    }
}
```

## Base Entity Pattern

```csharp
public abstract class BaseEntity
{
    public long Id { get; set; }
    public DateTime CreatedAt { get; set; }
    public DateTime? UpdatedAt { get; set; }
}

public abstract class AuditedEntity : BaseEntity
{
    public long? CreatedBy { get; set; }
}

public interface ISoftDeletable
{
    bool IsDeleted { get; set; }
    DateTime? DeletedAt { get; set; }
}

public class Product : AuditedEntity, ISoftDeletable
{
    public string Name { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public bool IsDeleted { get; set; }
    public DateTime? DeletedAt { get; set; }
    public long CategoryId { get; set; }
    public Category Category { get; set; } = null!;
}
```

## Query Patterns

### 1. IQueryable for Flexibility

```csharp
public IQueryable<Product> GetProducts(ProductFilter filter)
{
    var query = _context.Products.AsQueryable();

    if (filter.CategoryId.HasValue)
        query = query.Where(p => p.CategoryId == filter.CategoryId.Value);

    if (!string.IsNullOrWhiteSpace(filter.SearchTerm))
        query = query.Where(p => p.Name.Contains(filter.SearchTerm));

    return query; // Service adds sorting/paging
}
```

### 2. Projection with AutoMapper

```csharp
// ✅ Projects directly to DTO in SQL
var dtos = await _context.Products
    .ProjectTo<ProductListDto>(_mapper.ConfigurationProvider)
    .ToListAsync();
```

### 3. Pagination (⚡ Use ToPagedResultAsync at SERVICE Level)

**CRITICAL:** Pagination happens at **SERVICE layer**, NOT repository. Repositories return `IQueryable<T>`, services apply pagination.

```csharp
// ✅ Repository returns IQueryable
public IQueryable<Product> GetProducts(ProductFilter filter)
{
    var query = _dbSet.AsNoTracking();
    if (!string.IsNullOrEmpty(filter.SearchTerm))
        query = query.Where(p => p.Name.Contains(filter.SearchTerm));
    return query; // NO pagination
}

// ✅ Service applies pagination with ToPagedResultAsync
public async Task<PagedResult<ProductListDto>> GetProductsAsync(ProductFilter filter)
{
    var query = _productRepository.GetProducts(filter);
    query = query.OrderByDescending(p => p.CreatedAt);
    return await query.ToPagedResultAsync(filter.PageNumber, filter.PageSize);
}
```

### 4. Eager Loading

```csharp
// Single level
var product = await _context.Products
    .Include(p => p.Category)
    .Include(p => p.Supplier)
    .FirstOrDefaultAsync(p => p.Id == id);

// Multi-level (nested)
var order = await _context.Orders
    .Include(o => o.OrderItems).ThenInclude(i => i.Product)
    .Include(o => o.Customer)
    .FirstOrDefaultAsync(o => o.Id == id);
```

## Performance Patterns

### ⚡ CRITICAL: Always Filter at Database Level

The most common and costly performance mistake in Entity Framework is loading data into memory before filtering.

```csharp
// ❌ TERRIBLE PERFORMANCE - Loads ALL employees into memory, then filters
var query = _context.Employees
    .Include(e => e.Ranks.Where(r => r.IsActive))
    .Include(e => e.Assignments.Where(a => a.IsActive))
    .ToListAsync(); // Loads EVERYTHING!

var filtered = query.Result.Where(e =>
    e.Ranks.Any(r => r.RankId != 424) &&
    e.Assignments.Any(a => a.UnitId == unitId)
).Select(e => e.Id).ToList();

// ✅ OPTIMAL PERFORMANCE - Database filters before loading
var filtered = await _context.Employees
    .Where(e => e.Ranks.Any(r => r.IsActive && r.RankId != 424))
    .Where(e => e.Assignments.Any(a => a.IsActive && a.UnitId == unitId))
    .Select(e => e.Id)
    .ToListAsync();
```

**Why This Matters:**
- 1st approach: Loads 10,000 employees → 50MB transfer → filters in C# → slow
- 2nd approach: Database filters → returns 50 IDs → 1KB transfer → fast

**Golden Rules:**
1. ✅ Use `.Where()` and `.Any()` BEFORE `.ToListAsync()` or `.ToArrayAsync()`
2. ✅ Use `.Select()` to project only needed columns
3. ✅ Let SQL Server do the heavy lifting - it's optimized for filtering
4. ❌ NEVER call `.ToListAsync()` early and filter in memory
5. ❌ NEVER use `.Include()` if you only need IDs (use subqueries with `.Any()` instead)

### 1. AsNoTracking for Read-Only Queries

**Impact:** 30-50% faster for large queries (1K+ rows), 30-40% less memory. Tracking adds ~400 bytes overhead per entity.

```csharp
// ✅ BEST - Apply AsNoTracking early in query chain
var query = _dbSet
    .Include(s => s.Signature)
    .Include(s => s.Approver)
    .AsNoTracking()  // ⚡ Applies to all subsequent operations
    .Where(s => s.IsActive);

var count = await query.CountAsync();  // Benefits from AsNoTracking
var data = await query.ToListAsync();   // Benefits from AsNoTracking

// ❌ BAD - Tracking adds overhead for read-only scenarios
var products = await _context.Products
    .Where(p => p.Status == ProductStatus.Active)
    .ToListAsync();
```

**When to Use:**
- ✅ Read-only queries, API endpoints, reports, bulk exports
- ❌ Don't use if updating entities after loading

### 2. Batch Operations

```csharp
// ✅ Bulk insert
var products = new List<Product> { /* ... */ };
await _context.Products.AddRangeAsync(products);
await _context.SaveChangesAsync();

// ✅ Bulk update
var orders = await _context.Orders
    .Where(o => o.Status == OrderStatus.Pending)
    .ToListAsync();

foreach (var order in orders)
    order.Status = OrderStatus.Processing;

await _context.SaveChangesAsync(); // Single database round-trip
```

## Soft Delete Pattern

```csharp
public virtual async Task DeleteAsync(long id)
{
    var entity = await GetByIdAsync(id);
    if (entity == null) return;

    if (entity is ISoftDeletable softDeletable)
    {
        softDeletable.IsDeleted = true;
        softDeletable.DeletedAt = DateTime.UtcNow;
        _context.Entry(entity).State = EntityState.Modified;
    }
    else
        _context.Set<T>().Remove(entity);

    await _context.SaveChangesAsync();
}
```

## Transaction Handling

### 1. Implicit Transactions (Recommended)

```csharp
// SaveChangesAsync automatically wraps in transaction
public async Task<bool> CreateOrderWithItemsAsync(CreateOrderDto dto)
{
    var order = _mapper.Map<Order>(dto);
    await _context.Orders.AddAsync(order);

    var orderItems = CreateOrderItems(order, dto.Items);
    await _context.OrderItems.AddRangeAsync(orderItems);

    await _context.SaveChangesAsync(); // All or nothing
    return true;
}
```

### 2. Explicit Transactions

```csharp
public async Task<bool> ComplexOperationAsync()
{
    using var transaction = await _context.Database.BeginTransactionAsync();

    try
    {
        // Operation 1
        await _context.Orders.AddAsync(new Order { /* ... */ });
        await _context.SaveChangesAsync();

        // Operation 2
        await _context.OrderItems.AddAsync(new OrderItem { /* ... */ });
        await _context.SaveChangesAsync();

        // External call
        await _externalService.NotifyAsync();

        await transaction.CommitAsync();
        return true;
    }
    catch
    {
        await transaction.RollbackAsync();
        throw;
    }
}
```

## Common Pitfalls

### 0. Loading Data Then Filtering (MOST CRITICAL)

```csharp
// ❌ CRITICAL ERROR - Loads entire table into memory
var allEmployees = await _context.Employees
    .Include(e => e.Ranks)
    .Include(e => e.Assignments)
    .ToListAsync(); // STOP! This loads everything!

var activeEmployees = allEmployees
    .Where(e => e.IsActive)
    .Where(e => e.Ranks.Any(r => r.IsActive))
    .ToList();

// ✅ CORRECT - Filter at database level
var activeEmployees = await _context.Employees
    .Where(e => e.IsActive)
    .Where(e => e.Ranks.Any(r => r.IsActive))
    .ToListAsync();
```

**Impact:** This single mistake can make a query 100x-1000x slower!

**Always ask yourself:**
- Am I calling `.ToListAsync()` too early?
- Can this `.Where()` be moved before `.ToListAsync()`?
- Do I need `.Include()` or can I use `.Any()` for subqueries?

### 1. N+1 Query Problem

```csharp
// ❌ BAD - N+1 queries
var products = await _context.Products.ToListAsync();
foreach (var product in products)
{
    var category = await _context.Categories.FindAsync(product.CategoryId); // N queries!
}

// ✅ GOOD - Single query with Include
var products = await _context.Products
    .Include(p => p.Category)
    .ToListAsync();
```

### 2. Tracking Too Many Entities

```csharp
// ❌ BAD - Loads and tracks 10,000 entities in memory
var allProducts = await _context.Products.ToListAsync();

// ✅ GOOD - Process in batches
var pageSize = 100;
var pageNumber = 0;
List<Product> batch;

do
{
    batch = await _context.Products
        .AsNoTracking()
        .Skip(pageNumber * pageSize)
        .Take(pageSize)
        .ToListAsync();

    // Process batch
    await ProcessProducts(batch);

    pageNumber++;
} while (batch.Count == pageSize);
```

### 3. Lazy Loading Issues

```csharp
// ❌ Avoid lazy loading - causes unpredictable queries and N+1 problems
// Only enable if you fully understand the implications

// ✅ Use explicit eager loading instead
var products = await _context.Products
    .Include(p => p.Category)
    .Include(p => p.Supplier)
    .ToListAsync();
```

### 4. Improper Dispose Patterns

```csharp
// ❌ BAD - Manually managing DbContext
public class ProductService
{
    private readonly ApplicationDbContext _context = new ApplicationDbContext();
}

// ✅ GOOD - Let DI manage lifetime (Scoped)
public class ProductService
{
    private readonly ApplicationDbContext _context;

    public ProductService(ApplicationDbContext context)
    {
        _context = context;
    }
}
```

### 5. Not Using Async Properly

```csharp
// ❌ BAD - Blocking async calls
var product = _context.Products.FirstOrDefaultAsync(p => p.Id == id).Result;

// ✅ GOOD - Await async calls
var product = await _context.Products.FirstOrDefaultAsync(p => p.Id == id);
```

## Advanced Patterns

### Unit of Work Pattern

```csharp
public interface IUnitOfWork : IDisposable
{
    IProductRepository Products { get; }
    IOrderRepository Orders { get; }
    Task<int> CompleteAsync();
}

public class UnitOfWork : IUnitOfWork
{
    private readonly ApplicationDbContext _context;
    public IProductRepository Products { get; }
    public IOrderRepository Orders { get; }

    public UnitOfWork(ApplicationDbContext context)
    {
        _context = context;
        Products = new ProductRepository(_context);
        Orders = new OrderRepository(_context);
    }

    public async Task<int> CompleteAsync() => await _context.SaveChangesAsync();
    public void Dispose() => _context.Dispose();
}
```

## See Also

- [services-and-repositories.md](services-and-repositories.md) - Repository pattern implementation
- [architecture-overview.md](architecture-overview.md) - Clean architecture overview
- [dependency-injection.md](dependency-injection.md) - DI configuration for DbContext
- [complete-examples.md](complete-examples.md) - Full feature examples
