# Dependency Injection in .NET Core 8

## Table of Contents
- [Overview](#overview)
- [Service Lifetimes](#service-lifetimes)
- [Registration Patterns](#registration-patterns)
- [Constructor Injection](#constructor-injection)
- [Authentication & Authorization DI](#authentication--authorization-di)
- [HttpClient Registration](#httpclient-registration)
- [Configuration Injection](#configuration-injection)
- [Service Registration Validation](#service-registration-validation)
- [Common Patterns](#common-patterns)
- [Testing with DI](#testing-with-di)
- [Common Mistakes](#common-mistakes)
- [See Also](#see-also)

---

## Overview

.NET Core has built-in dependency injection (DI) that follows the **Dependency Inversion Principle** (the "D" in SOLID).

## Service Lifetimes

### 1. Transient

**When**: New instance every time it's requested

```csharp
builder.Services.AddTransient<IEmailService, EmailService>();
```

**Use for**:
- Lightweight, stateless services
- Services that don't share state
- Short-lived operations

### 2. Scoped

**When**: One instance per HTTP request

```csharp
builder.Services.AddScoped<IProductService, ProductService>();
builder.Services.AddScoped<IOrderRepository, OrderRepository>();
```

**Use for**:
- Repository classes
- Business logic services
- DbContext (most common)
- Services that should share state within a request

### 3. Singleton

**When**: Single instance for the application lifetime

```csharp
builder.Services.AddSingleton<ICacheService, CacheService>();
```

**Use for**:
- Expensive-to-create objects
- Thread-safe services
- Configuration objects
- Logging services

## Registration Patterns

### 1. Clean Architecture DI Setup

```csharp
// Program.cs - Extension method pattern
var builder = WebApplication.CreateBuilder(args);

// Add layers
builder.AddInfrastructure();
builder.AddApplication();
builder.AddPresentation();

var app = builder.Build();
app.Run();
```

### 2. Infrastructure Layer Registration

```csharp
// ProgramConfig.cs or InfrastructureServiceRegistration.cs
public static class InfrastructureExtensions
{
    public static WebApplicationBuilder AddInfrastructure(this WebApplicationBuilder builder)
    {
        // DbContext
        builder.Services.AddDbContext<ApplicationDbContext>(options =>
            options.UseSqlServer(
                builder.Configuration.GetConnectionString("DefaultConnection"),
                b => b.MigrationsAssembly("YourProject.Infrastructure")
            )
        );

        // Generic Repository
        builder.Services.AddScoped(typeof(IRepository<>), typeof(Repository<>));

        // Specific Repositories
        builder.Services.AddScoped<IProductRepository, ProductRepository>();
        builder.Services.AddScoped<IOrderRepository, OrderRepository>();
        builder.Services.AddScoped<ICustomerRepository, CustomerRepository>();

        // External Services
        builder.Services.AddHttpClient<IExternalApiService, ExternalApiService>();

        return builder;
    }
}
```

### 3. Application Layer Registration

```csharp
public static class ApplicationExtensions
{
    public static WebApplicationBuilder AddApplication(this WebApplicationBuilder builder)
    {
        // AutoMapper
        builder.Services.AddAutoMapper(typeof(ProductMappingProfile).Assembly);

        // Services
        builder.Services.AddScoped<IProductService, ProductService>();
        builder.Services.AddScoped<IOrderService, OrderService>();
        builder.Services.AddScoped<ICustomerService, CustomerService>();

        // MediatR (if using)
        builder.Services.AddMediatR(cfg =>
            cfg.RegisterServicesFromAssembly(typeof(CreateProductCommand).Assembly)
        );

        // FluentValidation
        builder.Services.AddValidatorsFromAssembly(typeof(CreateProductDtoValidator).Assembly);

        return builder;
    }
}
```

### 4. Presentation Layer Registration

```csharp
public static class PresentationExtensions
{
    public static WebApplicationBuilder AddPresentation(this WebApplicationBuilder builder)
    {
        // Controllers
        builder.Services.AddControllers()
            .AddJsonOptions(options =>
            {
                options.JsonSerializerOptions.ReferenceHandler = ReferenceHandler.IgnoreCycles;
                options.JsonSerializerOptions.DefaultIgnoreCondition = JsonIgnoreCondition.WhenWritingNull;
            });

        // API Versioning
        builder.Services.AddApiVersioning(options =>
        {
            options.DefaultApiVersion = new ApiVersion(1, 0);
            options.AssumeDefaultVersionWhenUnspecified = true;
            options.ReportApiVersions = true;
        });

        // Swagger
        builder.Services.AddEndpointsApiExplorer();
        builder.Services.AddSwaggerGen(c =>
        {
            c.SwaggerDoc("v1", new OpenApiInfo { Title = "Your API", Version = "v1" });
        });

        // CORS
        builder.Services.AddCors(options =>
        {
            options.AddPolicy("AllowAll", policy =>
            {
                policy.AllowAnyOrigin()
                      .AllowAnyMethod()
                      .AllowAnyHeader();
            });
        });

        // Output Caching
        builder.Services.AddOutputCache(options =>
        {
            options.AddBasePolicy(builder => builder.Expire(TimeSpan.FromMinutes(1)));
            options.AddPolicy("ProductList", builder => builder.Expire(TimeSpan.FromMinutes(5)));
        });

        // Rate Limiting
        builder.Services.AddRateLimiter(options =>
        {
            options.GlobalLimiter = PartitionedRateLimiter.Create<HttpContext, string>(context =>
                RateLimitPartition.GetFixedWindowLimiter(
                    partitionKey: context.User.Identity?.Name ?? context.Request.Headers.Host.ToString(),
                    factory: partition => new FixedWindowRateLimiterOptions
                    {
                        AutoReplenishment = true,
                        PermitLimit = 100,
                        QueueLimit = 0,
                        Window = TimeSpan.FromMinutes(1)
                    }));
        });

        return builder;
    }
}
```

## Constructor Injection

### Basic Pattern

```csharp
public class ProductController : ControllerBase
{
    private readonly IProductService _productService;
    private readonly ILogger<ProductController> _logger;

    // Constructor injection
    public ProductController(
        IProductService productService,
        ILogger<ProductController> logger)
    {
        _productService = productService;
        _logger = logger;
    }

    [HttpGet]
    public async Task<ActionResult<List<ProductDto>>> GetProducts()
    {
        var products = await _productService.GetAllAsync();
        return Ok(products);
    }
}
```

### Primary Constructor (C# 12+)

```csharp
public class ProductService(
    IProductRepository productRepository,
    IMapper mapper,
    ILogger<ProductService> logger) : IProductService
{
    // Fields automatically created from parameters

    public async Task<List<ProductDto>> GetAllAsync()
    {
        var products = await productRepository.GetAllAsync();
        return mapper.Map<List<ProductDto>>(products);
    }
}
```

## Authentication & Authorization DI

### JWT Bearer Authentication

```csharp
public static WebApplicationBuilder AddAuthentication(this WebApplicationBuilder builder)
{
    var jwtSettings = builder.Configuration.GetSection("JwtSettings");

    builder.Services.AddAuthentication(options =>
    {
        options.DefaultAuthenticateScheme = JwtBearerDefaults.AuthenticationScheme;
        options.DefaultChallengeScheme = JwtBearerDefaults.AuthenticationScheme;
    })
    .AddJwtBearer(options =>
    {
        options.TokenValidationParameters = new TokenValidationParameters
        {
            ValidateIssuer = true,
            ValidateAudience = true,
            ValidateLifetime = true,
            ValidateIssuerSigningKey = true,
            ValidIssuer = jwtSettings["Issuer"],
            ValidAudience = jwtSettings["Audience"],
            IssuerSigningKey = new SymmetricSecurityKey(
                Encoding.UTF8.GetBytes(jwtSettings["SecretKey"]!))
        };
    });

    return builder;
}
```

### Authorization Policies

```csharp
public static WebApplicationBuilder AddAuthorizationPolicies(this WebApplicationBuilder builder)
{
    builder.Services.AddAuthorization(options =>
    {
        options.AddPolicy("AdminOnly", policy =>
            policy.RequireRole("Admin"));

        options.AddPolicy("CanViewProducts", policy =>
            policy.RequireClaim("permissions", "products:read"));

        options.AddPolicy("CanEditProducts", policy =>
            policy.RequireClaim("permissions", "products:write"));
    });

    return builder;
}
```

## HttpClient Registration

### Named Clients

```csharp
builder.Services.AddHttpClient("ExternalAPI", client =>
{
    client.BaseAddress = new Uri("https://api.external.com/");
    client.DefaultRequestHeaders.Add("Accept", "application/json");
    client.Timeout = TimeSpan.FromSeconds(30);
});
```

### Typed Clients

```csharp
builder.Services.AddHttpClient<IExternalApiService, ExternalApiService>(client =>
{
    client.BaseAddress = new Uri("https://api.external.com/");
    client.DefaultRequestHeaders.Add("Accept", "application/json");
})
.AddPolicyHandler(GetRetryPolicy())
.AddPolicyHandler(GetCircuitBreakerPolicy());

static IAsyncPolicy<HttpResponseMessage> GetRetryPolicy()
{
    return HttpPolicyExtensions
        .HandleTransientHttpError()
        .WaitAndRetryAsync(3, retryAttempt => TimeSpan.FromSeconds(Math.Pow(2, retryAttempt)));
}

static IAsyncPolicy<HttpResponseMessage> GetCircuitBreakerPolicy()
{
    return HttpPolicyExtensions
        .HandleTransientHttpError()
        .CircuitBreakerAsync(5, TimeSpan.FromSeconds(30));
}
```

## Configuration Injection

### Options Pattern

```csharp
// appsettings.json
{
  "EmailSettings": {
    "SmtpServer": "smtp.gmail.com",
    "Port": 587,
    "Username": "your-email@gmail.com"
  }
}

// Configuration class
public class EmailSettings
{
    public string SmtpServer { get; set; } = string.Empty;
    public int Port { get; set; }
    public string Username { get; set; } = string.Empty;
}

// Registration
builder.Services.Configure<EmailSettings>(
    builder.Configuration.GetSection("EmailSettings"));

// Injection
public class EmailService(IOptions<EmailSettings> emailSettings) : IEmailService
{
    private readonly EmailSettings _settings = emailSettings.Value;

    public async Task SendEmailAsync(string to, string subject, string body)
    {
        // Use _settings.SmtpServer, _settings.Port, etc.
    }
}
```

## Service Registration Validation

### Validate on Startup

```csharp
var app = builder.Build();

// Validate all scoped services can be created
using (var scope = app.Services.CreateScope())
{
    var services = scope.ServiceProvider;

    try
    {
        var dbContext = services.GetRequiredService<ApplicationDbContext>();
        var productService = services.GetRequiredService<IProductService>();
        // ... validate other critical services
    }
    catch (Exception ex)
    {
        var logger = services.GetRequiredService<ILogger<Program>>();
        logger.LogError(ex, "An error occurred while validating services.");
        throw;
    }
}

app.Run();
```

## Common Patterns

### Factory Pattern

```csharp
public interface IServiceFactory
{
    IPaymentService CreatePaymentService(PaymentType type);
}

public class ServiceFactory(IServiceProvider serviceProvider) : IServiceFactory
{
    public IPaymentService CreatePaymentService(PaymentType type)
    {
        return type switch
        {
            PaymentType.CreditCard => serviceProvider.GetRequiredService<CreditCardPaymentService>(),
            PaymentType.PayPal => serviceProvider.GetRequiredService<PayPalPaymentService>(),
            _ => throw new ArgumentException("Invalid payment type")
        };
    }
}

// Registration
builder.Services.AddScoped<IServiceFactory, ServiceFactory>();
builder.Services.AddScoped<CreditCardPaymentService>();
builder.Services.AddScoped<PayPalPaymentService>();
```

### Decorator Pattern

```csharp
// Base service
builder.Services.AddScoped<IProductService, ProductService>();

// Decorator
builder.Services.Decorate<IProductService, CachedProductService>();

// CachedProductService implementation
public class CachedProductService(
    IProductService innerService,
    IMemoryCache cache) : IProductService
{
    public async Task<ProductDto?> GetByIdAsync(long id)
    {
        var cacheKey = $"product_{id}";

        if (cache.TryGetValue(cacheKey, out ProductDto? cachedProduct))
            return cachedProduct;

        var product = await innerService.GetByIdAsync(id);

        if (product != null)
            cache.Set(cacheKey, product, TimeSpan.FromMinutes(5));

        return product;
    }
}
```

## Testing with DI

### Mock Services in Tests

```csharp
public class ProductServiceTests
{
    private readonly Mock<IProductRepository> _mockRepository;
    private readonly Mock<IMapper> _mockMapper;
    private readonly ProductService _service;

    public ProductServiceTests()
    {
        _mockRepository = new Mock<IProductRepository>();
        _mockMapper = new Mock<IMapper>();
        _service = new ProductService(_mockRepository.Object, _mockMapper.Object);
    }

    [Fact]
    public async Task GetByIdAsync_ReturnsProduct()
    {
        // Arrange
        var product = new Product { Id = 1, Name = "Test Product" };
        _mockRepository.Setup(r => r.GetByIdAsync(1)).ReturnsAsync(product);

        // Act
        var result = await _service.GetByIdAsync(1);

        // Assert
        Assert.NotNull(result);
        Assert.Equal(1, result.Id);
    }
}
```

## Common Mistakes

### ❌ Using ServiceLocator Anti-Pattern

```csharp
// WRONG - Service Locator
public class ProductController : ControllerBase
{
    private readonly IServiceProvider _serviceProvider;

    public ProductController(IServiceProvider serviceProvider)
    {
        _serviceProvider = serviceProvider;
    }

    public async Task<IActionResult> Get()
    {
        var service = _serviceProvider.GetRequiredService<IProductService>(); // ❌
        return Ok(await service.GetAllAsync());
    }
}
```

### ✅ Proper Constructor Injection

```csharp
// CORRECT
public class ProductController(IProductService productService) : ControllerBase
{
    [HttpGet]
    public async Task<IActionResult> Get()
    {
        return Ok(await productService.GetAllAsync());
    }
}
```

### ❌ Wrong Lifetime

```csharp
// WRONG - DbContext as Singleton (causes thread safety issues)
builder.Services.AddSingleton<ApplicationDbContext>();

// CORRECT - DbContext as Scoped
builder.Services.AddDbContext<ApplicationDbContext>(options => /* ... */);
```

### ❌ Captive Dependencies

```csharp
// WRONG - Singleton depending on Scoped service
builder.Services.AddSingleton<MyCacheService>(); // Singleton
builder.Services.AddScoped<ApplicationDbContext>(); // Scoped

public class MyCacheService(ApplicationDbContext context) // ❌ Captures scoped in singleton!
{
}

// CORRECT - Match lifetimes or use IServiceScopeFactory
public class MyCacheService(IServiceScopeFactory scopeFactory)
{
    public async Task<Product?> GetProductAsync(long id)
    {
        using var scope = scopeFactory.CreateScope();
        var context = scope.ServiceProvider.GetRequiredService<ApplicationDbContext>();
        return await context.Products.FindAsync(id);
    }
}
```

## See Also

- [architecture-overview.md](architecture-overview.md) - Clean architecture structure
- [services-and-repositories.md](services-and-repositories.md) - Service and repository patterns
- [testing-guide.md](testing-guide.md) - Testing with DI
