# Controllers and Routing - Best Practices

Complete guide to clean API controller definitions and routing patterns in ASP.NET Core.

## Table of Contents
- [The Golden Rule](#the-golden-rule)
- [Clean Controller Pattern](#clean-controller-pattern)
- [Routing Patterns](#routing-patterns)
- [Action Results](#action-results)
- [Model Binding](#model-binding)
- [Authorization](#authorization)
- [Output Caching](#output-caching)
- [Versioning](#versioning)
- [Best Practices](#best-practices)
- [Summary](#summary)

---

## The Golden Rule

**Controllers should ONLY:**
- ✅ Handle HTTP request/response
- ✅ Validate model state
- ✅ Delegate to services
- ✅ Return appropriate HTTP status codes

**Controllers should NEVER:**
- ❌ Contain business logic
- ❌ Access database directly
- ❌ Implement complex validation logic
- ❌ Handle exceptions (use middleware)

---

## Clean Controller Pattern

### Basic Controller Structure

```csharp
using Microsoft.AspNetCore.Mvc;

namespace MyApi.Controllers;

[ApiController]
[Route("api/[controller]")]
public class UsersController : ControllerBase
{
    private readonly IUserService _userService;
    private readonly ILogger<UsersController> _logger;

    public UsersController(
        IUserService userService,
        ILogger<UsersController> logger)
    {
        _userService = userService;
        _logger = logger;
    }

    // GET: api/users
    [HttpGet]
    [ProducesResponseType(typeof(IEnumerable<UserResponseDto>), StatusCodes.Status200OK)]
    public async Task<IActionResult> GetUsers()
    {
        var users = await _userService.GetAllUsersAsync();
        return Ok(users);
    }

    // GET: api/users/5
    [HttpGet("{id}")]
    [ProducesResponseType(typeof(UserResponseDto), StatusCodes.Status200OK)]
    [ProducesResponseType(StatusCodes.Status404NotFound)]
    public async Task<IActionResult> GetUser(int id)
    {
        var user = await _userService.GetUserByIdAsync(id);

        if (user == null)
            return NotFound();

        return Ok(user);
    }

    // POST: api/users
    [HttpPost]
    [ProducesResponseType(typeof(UserResponseDto), StatusCodes.Status201Created)]
    [ProducesResponseType(StatusCodes.Status400BadRequest)]
    public async Task<IActionResult> CreateUser([FromBody] CreateUserDto dto)
    {
        if (!ModelState.IsValid)
            return BadRequest(ModelState);

        var user = await _userService.CreateUserAsync(dto);

        return CreatedAtAction(
            nameof(GetUser),
            new { id = user.Id },
            user);
    }

    // PUT: api/users/5
    [HttpPut("{id}")]
    [ProducesResponseType(StatusCodes.Status204NoContent)]
    [ProducesResponseType(StatusCodes.Status400BadRequest)]
    [ProducesResponseType(StatusCodes.Status404NotFound)]
    public async Task<IActionResult> UpdateUser(int id, [FromBody] UpdateUserDto dto)
    {
        if (!ModelState.IsValid)
            return BadRequest(ModelState);

        var success = await _userService.UpdateUserAsync(id, dto);

        if (!success)
            return NotFound();

        return NoContent();
    }

    // DELETE: api/users/5
    [HttpDelete("{id}")]
    [ProducesResponseType(StatusCodes.Status204NoContent)]
    [ProducesResponseType(StatusCodes.Status404NotFound)]
    public async Task<IActionResult> DeleteUser(int id)
    {
        var success = await _userService.DeleteUserAsync(id);

        if (!success)
            return NotFound();

        return NoContent();
    }
}
```

**Key Points:**
- `[ApiController]` attribute for automatic model validation
- `[Route]` attribute for routing configuration
- Dependency injection in constructor
- Async methods for all I/O operations
- Proper HTTP status codes
- `ProducesResponseType` for Swagger documentation

---

## Routing Patterns

### Attribute Routing (RECOMMENDED)

```csharp
[ApiController]
[Route("api/[controller]")]
public class ProductsController : ControllerBase
{
    // GET: api/products
    [HttpGet]
    public async Task<IActionResult> GetProducts() { }

    // GET: api/products/5
    [HttpGet("{id}")]
    public async Task<IActionResult> GetProduct(int id) { }

    // GET: api/products/category/electronics
    [HttpGet("category/{categoryName}")]
    public async Task<IActionResult> GetByCategory(string categoryName) { }

    // GET: api/products/search?query=laptop
    [HttpGet("search")]
    public async Task<IActionResult> Search([FromQuery] string query) { }

    // POST: api/products
    [HttpPost]
    public async Task<IActionResult> CreateProduct([FromBody] CreateProductDto dto) { }

    // PUT: api/products/5
    [HttpPut("{id}")]
    public async Task<IActionResult> UpdateProduct(int id, [FromBody] UpdateProductDto dto) { }

    // DELETE: api/products/5
    [HttpDelete("{id}")]
    public async Task<IActionResult> DeleteProduct(int id) { }
}
```

### Route Constraints

```csharp
// Only integers
[HttpGet("{id:int}")]
public async Task<IActionResult> GetProduct(int id) { }

// Only GUIDs
[HttpGet("{id:guid}")]
public async Task<IActionResult> GetProduct(Guid id) { }

// Minimum value
[HttpGet("page/{pageNumber:min(1)}")]
public async Task<IActionResult> GetPage(int pageNumber) { }

// Regex pattern
[HttpGet("{email:regex(^[a-zA-Z0-9@.]+$)}")]
public async Task<IActionResult> GetByEmail(string email) { }
```

---

## Action Results

### Common Return Types

```csharp
// 200 OK with data
return Ok(data);

// 201 Created with location header
return CreatedAtAction(nameof(GetUser), new { id = user.Id }, user);

// 204 No Content
return NoContent();

// 400 Bad Request
return BadRequest("Invalid input");
return BadRequest(ModelState);

// 401 Unauthorized
return Unauthorized();

// 403 Forbidden
return Forbid();

// 404 Not Found
return NotFound();
return NotFound("User not found");

// 500 Internal Server Error (handled by middleware usually)
return StatusCode(500, "Internal server error");
```

### Generic ActionResult<T>

```csharp
[HttpGet("{id}")]
public async Task<ActionResult<UserResponseDto>> GetUser(int id)
{
    var user = await _userService.GetUserByIdAsync(id);

    if (user == null)
        return NotFound();

    return user; // Implicit conversion to Ok(user)
}
```

---

## Model Binding

### From Body

```csharp
[HttpPost]
public async Task<IActionResult> CreateUser([FromBody] CreateUserDto dto)
{
    // dto populated from request body (JSON)
}
```

### From Route

```csharp
[HttpGet("{id}")]
public async Task<IActionResult> GetUser([FromRoute] int id)
{
    // id from route parameter
}
```

### From Query

```csharp
[HttpGet]
public async Task<IActionResult> Search(
    [FromQuery] string query,
    [FromQuery] int page = 1,
    [FromQuery] int pageSize = 10)
{
    // Parameters from query string: ?query=test&page=1&pageSize=10
}
```

### From Header

```csharp
[HttpGet]
public async Task<IActionResult> GetData([FromHeader(Name = "X-Api-Key")] string apiKey)
{
    // Value from request header
}
```

### From Services (Dependency Injection)

```csharp
[HttpGet]
public async Task<IActionResult> GetData([FromServices] ISpecialService service)
{
    // Inject service for this action only
}
```

---

## Authorization

### Authorize Attribute

```csharp
// Require authentication for entire controller
[Authorize]
[ApiController]
[Route("api/[controller]")]
public class UsersController : ControllerBase
{
    // All actions require authentication
}

// Require specific role
[Authorize(Roles = "Admin")]
[HttpDelete("{id}")]
public async Task<IActionResult> DeleteUser(int id) { }

// Require specific policy
[Authorize(Policy = "CanEditUsers")]
[HttpPut("{id}")]
public async Task<IActionResult> UpdateUser(int id, UpdateUserDto dto) { }

// Allow anonymous access to specific action
[AllowAnonymous]
[HttpPost("login")]
public async Task<IActionResult> Login(LoginDto dto) { }
```

---

## Output Caching

### Adding Cache Policies to Endpoints

**⚠️ CRITICAL REMINDER:** When adding a new endpoint with `[OutputCache]` attribute, you MUST also add the corresponding cache policy to the project's cache configuration file.

```csharp
// Step 1: Add cache attribute to endpoint
[HttpGet("employee/{employeeId}/active-role-name")]
[OutputCache(PolicyName = "InternalActiveRoleName")]
[ProducesResponseType(typeof(ApiResponse<string>), 200)]
public async Task<IActionResult> GetActiveRoleNameByEmployeeId(long employeeId)
{
    var roleName = await _employeeRoleService.GetActiveRoleNameByEmployeeId(employeeId);
    if (roleName == null)
        return NotFound();

    return Ok(roleName);
}
```

```csharp
// Step 2: Add policy to CachePolicyConfiguration.cs
// Example location: YourProject.API/Configuration/CachePolicyConfiguration.cs

private static void ConfigureInternalRolePolicies(OutputCacheOptions options)
{
    options.AddPolicy("InternalActiveRoleName", b => b
        .Expire(Medium)  // Cache duration
        .SetVaryByRouteValue("employeeId"));  // Vary by route parameter
}
```

### Cache Policy Patterns

```csharp
// Cache with route parameter variation
options.AddPolicy("InternalEmployeeById", b => b
    .Expire(Medium)
    .SetVaryByRouteValue("id"));

// Cache with query parameter variation
options.AddPolicy("InternalEmployeesPaged", b => b
    .Expire(Short)
    .SetVaryByQuery("pageNumber", "pageSize"));

// Cache with multiple variations
options.AddPolicy("InternalOrgUnitsFiltered", b => b
    .Expire(Long)
    .SetVaryByQuery("searchTerm", "type", "pageNumber", "pageSize", "parentId"));

// Cache with header variation (e.g., for user-specific data)
options.AddPolicy("CurrentEmployee", b => b
    .Expire(Short)
    .SetCacheKeyPrefix("me")
    .SetVaryByHeader("Authorization"));
```

### Common Cache Durations

```csharp
// Define in CachePolicyConfiguration.cs
private static readonly TimeSpan VeryShort = TimeSpan.FromSeconds(10);    // 10s
private static readonly TimeSpan Short = TimeSpan.FromSeconds(30);        // 30s
private static readonly TimeSpan Medium = TimeSpan.FromSeconds(60);       // 60s
private static readonly TimeSpan Long = TimeSpan.FromMinutes(2);          // 2min
private static readonly TimeSpan VeryLong = TimeSpan.FromMinutes(5);      // 5min
```

### Cache Policy Guidelines

1. **Internal APIs** - Use shorter cache durations (Short/Medium) for service-to-service communication
2. **Public APIs** - Can use longer cache durations (Long/VeryLong) for stable data
3. **Fast Endpoints** - For optimized endpoints returning minimal data (like name-only), use Medium duration
4. **User-Specific Data** - Always vary by Authorization header
5. **Filtered/Paginated Data** - Vary by all relevant query parameters

**Common Mistake:** Forgetting to add the cache policy configuration will cause runtime errors when the endpoint is called.

---

## Versioning

### URL Versioning

```csharp
[ApiController]
[Route("api/v{version:apiVersion}/[controller]")]
[ApiVersion("1.0")]
public class UsersV1Controller : ControllerBase
{
    // api/v1/users
}

[ApiController]
[Route("api/v{version:apiVersion}/[controller]")]
[ApiVersion("2.0")]
public class UsersV2Controller : ControllerBase
{
    // api/v2/users
}
```

### Header Versioning

```csharp
// Program.cs
builder.Services.AddApiVersioning(options =>
{
    options.DefaultApiVersion = new ApiVersion(1, 0);
    options.AssumeDefaultVersionWhenUnspecified = true;
    options.ReportApiVersions = true;
    options.ApiVersionReader = new HeaderApiVersionReader("api-version");
});

// Controller
[ApiVersion("1.0")]
public class UsersController : ControllerBase
{
    // Request header: api-version: 1.0
}
```

---

## Best Practices

### 1. Keep Controllers Thin

```csharp
// ❌ BAD - Business logic in controller
[HttpPost]
public async Task<IActionResult> CreateOrder(CreateOrderDto dto)
{
    // Validate stock
    var product = await _context.Products.FindAsync(dto.ProductId);
    if (product.Stock < dto.Quantity)
        return BadRequest("Insufficient stock");

    // Calculate total
    var total = product.Price * dto.Quantity;

    // Apply discount
    if (dto.DiscountCode != null)
    {
        var discount = await _context.Discounts
            .FirstOrDefaultAsync(d => d.Code == dto.DiscountCode);
        if (discount != null)
            total -= total * discount.Percentage;
    }

    // Create order
    var order = new Order { Total = total, ProductId = dto.ProductId };
    _context.Orders.Add(order);
    await _context.SaveChangesAsync();

    return Ok(order);
}

// ✅ GOOD - Delegate to service
[HttpPost]
public async Task<IActionResult> CreateOrder(CreateOrderDto dto)
{
    var order = await _orderService.CreateOrderAsync(dto);
    return CreatedAtAction(nameof(GetOrder), new { id = order.Id }, order);
}
```

### 2. Use DTOs, Not Entities

```csharp
// ❌ BAD - Exposing entity directly
[HttpGet("{id}")]
public async Task<ActionResult<User>> GetUser(int id)
{
    return await _context.Users.FindAsync(id);
    // Exposes all properties, including sensitive data
}

// ✅ GOOD - Use DTO
[HttpGet("{id}")]
public async Task<ActionResult<UserResponseDto>> GetUser(int id)
{
    var user = await _userService.GetUserByIdAsync(id);
    return user; // UserResponseDto with only necessary properties
}
```

### 3. Validate Input

```csharp
[HttpPost]
public async Task<IActionResult> CreateUser([FromBody] CreateUserDto dto)
{
    // ModelState validation happens automatically with [ApiController]
    if (!ModelState.IsValid)
        return BadRequest(ModelState);

    // Additional business validation in service
    var user = await _userService.CreateUserAsync(dto);

    return CreatedAtAction(nameof(GetUser), new { id = user.Id }, user);
}
```

### 4. Use Async/Await

```csharp
// ✅ ALWAYS async for I/O operations
[HttpGet]
public async Task<IActionResult> GetUsers()
{
    var users = await _userService.GetAllUsersAsync();
    return Ok(users);
}

// ❌ NEVER block with .Result or .Wait()
[HttpGet]
public IActionResult GetUsers()
{
    var users = _userService.GetAllUsersAsync().Result; // DON'T DO THIS
    return Ok(users);
}
```

### 5. Document with Swagger

```csharp
/// <summary>
/// Gets a user by ID
/// </summary>
/// <param name="id">The user ID</param>
/// <returns>The user details</returns>
/// <response code="200">Returns the user</response>
/// <response code="404">If the user is not found</response>
[HttpGet("{id}")]
[ProducesResponseType(typeof(UserResponseDto), StatusCodes.Status200OK)]
[ProducesResponseType(StatusCodes.Status404NotFound)]
public async Task<IActionResult> GetUser(int id)
{
    var user = await _userService.GetUserByIdAsync(id);

    if (user == null)
        return NotFound();

    return Ok(user);
}
```

---

## Summary

**Controller Best Practices:**
1. **Thin Controllers** - Delegate to services
2. **DTOs** - Never expose entities directly
3. **Async/Await** - Always for I/O operations
4. **Validation** - Use `[ApiController]` and FluentValidation
5. **Status Codes** - Return appropriate HTTP codes
6. **Documentation** - Use XML comments and ProducesResponseType
7. **Authorization** - Apply `[Authorize]` where needed
8. **Error Handling** - Let middleware handle exceptions

**See Also:**
- [services-and-repositories.md](services-and-repositories.md) - Service layer patterns
- [validation-patterns.md](validation-patterns.md) - Input validation
- [complete-examples.md](complete-examples.md) - Full examples
