# Complete Feature Examples

This document shows complete, end-to-end implementations of common features following Clean Architecture principles.

## Table of Contents
- [Example 1: Product Management Feature](#example-1-product-management-feature)
- [Key Takeaways](#key-takeaways)
- [See Also](#see-also)

---

## Example 1: Product Management Feature

### Domain Layer (Core)

**Product.cs** (Entity)
```csharp
namespace YourProject.Core.Entities;

public class Product : AuditedEntity, ISoftDeletable
{
    public string Name { get; set; } = string.Empty;
    public string Sku { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public int Stock { get; set; }
    public ProductStatus Status { get; set; }
    public bool IsDeleted { get; set; }
    public DateTime? DeletedAt { get; set; }

    // Navigation properties
    public long CategoryId { get; set; }
    public Category Category { get; set; } = null!;

    public long? SupplierId { get; set; }
    public Supplier? Supplier { get; set; }

    public ICollection<OrderItem> OrderItems { get; set; } = new List<OrderItem>();
}

public enum ProductStatus
{
    Draft,
    Active,
    Discontinued
}
```

**IProductRepository.cs** (Interface)
```csharp
namespace YourProject.Core.Interfaces;

public interface IProductRepository : IRepository<Product>
{
    IQueryable<Product> GetActiveProducts(ProductFilter filter);
    Task<Product?> GetBySkuAsync(string sku);
    Task<bool> ExistsBySkuAsync(string sku);
    Task<List<Product>> GetLowStockProductsAsync(int threshold);
}
```

**ProductFilter.cs** (Filter Model)
```csharp
namespace YourProject.Core.Models;

public class ProductFilter
{
    public long? CategoryId { get; set; }
    public decimal? MinPrice { get; set; }
    public decimal? MaxPrice { get; set; }
    public ProductStatus? Status { get; set; }
    public string? SearchTerm { get; set; }
    public int PageNumber { get; set; } = 1;
    public int PageSize { get; set; } = 10;
}
```

### Infrastructure Layer

**ProductRepository.cs** (Implementation)
```csharp
namespace YourProject.Infrastructure.Repositories;

public class ProductRepository : Repository<Product>, IProductRepository
{
    public ProductRepository(ApplicationDbContext context) : base(context) { }

    public IQueryable<Product> GetActiveProducts(ProductFilter filter)
    {
        var query = _context.Products
            .Where(p => p.Status == ProductStatus.Active);

        if (filter.CategoryId.HasValue)
            query = query.Where(p => p.CategoryId == filter.CategoryId.Value);

        if (filter.MinPrice.HasValue)
            query = query.Where(p => p.Price >= filter.MinPrice.Value);

        if (filter.MaxPrice.HasValue)
            query = query.Where(p => p.Price <= filter.MaxPrice.Value);

        if (!string.IsNullOrWhiteSpace(filter.SearchTerm))
            query = query.Where(p => p.Name.Contains(filter.SearchTerm)
                                  || p.Sku.Contains(filter.SearchTerm));

        return query;
    }

    public async Task<Product?> GetBySkuAsync(string sku)
    {
        return await _context.Products
            .FirstOrDefaultAsync(p => p.Sku == sku);
    }

    public async Task<bool> ExistsBySkuAsync(string sku)
    {
        return await _context.Products
            .AnyAsync(p => p.Sku == sku);
    }

    public async Task<List<Product>> GetLowStockProductsAsync(int threshold)
    {
        return await _context.Products
            .Where(p => p.Stock <= threshold && p.Status == ProductStatus.Active)
            .Include(p => p.Category)
            .Include(p => p.Supplier)
            .ToListAsync();
    }
}
```

**ProductConfiguration.cs** (EF Core Configuration)
```csharp
namespace YourProject.Infrastructure.Data.Configurations;

public class ProductConfiguration : IEntityTypeConfiguration<Product>
{
    public void Configure(EntityTypeBuilder<Product> builder)
    {
        builder.ToTable("Products");

        builder.HasKey(p => p.Id);

        builder.Property(p => p.Name)
            .IsRequired()
            .HasMaxLength(200);

        builder.Property(p => p.Sku)
            .IsRequired()
            .HasMaxLength(50);

        builder.Property(p => p.Price)
            .HasColumnType("decimal(18,2)");

        builder.Property(p => p.Status)
            .HasConversion<string>()
            .HasMaxLength(50);

        builder.HasIndex(p => p.Sku).IsUnique();
        builder.HasIndex(p => new { p.CategoryId, p.Status });

        builder.HasOne(p => p.Category)
            .WithMany(c => c.Products)
            .HasForeignKey(p => p.CategoryId)
            .OnDelete(DeleteBehavior.Restrict);

        builder.HasOne(p => p.Supplier)
            .WithMany(s => s.Products)
            .HasForeignKey(p => p.SupplierId)
            .OnDelete(DeleteBehavior.SetNull);
    }
}
```

### Application Layer

**DTOs**
```csharp
namespace YourProject.Application.DTOs;

public class ProductListDto
{
    public long Id { get; set; }
    public string Name { get; set; } = string.Empty;
    public string Sku { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public int Stock { get; set; }
    public string Status { get; set; } = string.Empty;
    public string CategoryName { get; set; } = string.Empty;
}

public class ProductDetailsDto : ProductListDto
{
    public string? SupplierName { get; set; }
    public DateTime CreatedAt { get; set; }
    public DateTime? UpdatedAt { get; set; }
}

public class CreateProductDto
{
    public string Name { get; set; } = string.Empty;
    public string Sku { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public int Stock { get; set; }
    public long CategoryId { get; set; }
    public long? SupplierId { get; set; }
}

public class UpdateProductDto
{
    public string Name { get; set; } = string.Empty;
    public decimal Price { get; set; }
    public int Stock { get; set; }
    public ProductStatus Status { get; set; }
}
```

**Validators**
```csharp
namespace YourProject.Application.Validators;

public class CreateProductDtoValidator : AbstractValidator<CreateProductDto>
{
    private readonly IProductRepository _productRepository;

    public CreateProductDtoValidator(IProductRepository productRepository)
    {
        _productRepository = productRepository;

        RuleFor(x => x.Name)
            .NotEmpty().WithMessage("Product name is required")
            .Length(3, 200).WithMessage("Name must be between 3 and 200 characters");

        RuleFor(x => x.Sku)
            .NotEmpty().WithMessage("SKU is required")
            .MaximumLength(50).WithMessage("SKU cannot exceed 50 characters")
            .Matches(@"^[A-Z0-9-]+$").WithMessage("SKU can only contain uppercase letters, numbers, and hyphens")
            .MustAsync(BeUniqueSku).WithMessage("SKU already exists");

        RuleFor(x => x.Price)
            .GreaterThan(0).WithMessage("Price must be greater than 0")
            .LessThanOrEqualTo(1000000).WithMessage("Price cannot exceed 1,000,000");

        RuleFor(x => x.Stock)
            .GreaterThanOrEqualTo(0).WithMessage("Stock cannot be negative");

        RuleFor(x => x.CategoryId)
            .GreaterThan(0).WithMessage("Category is required");
    }

    private async Task<bool> BeUniqueSku(string sku, CancellationToken cancellationToken)
    {
        return !await _productRepository.ExistsBySkuAsync(sku);
    }
}
```

**AutoMapper Profile**
```csharp
namespace YourProject.Application.Mapping;

public class ProductMappingProfile : Profile
{
    public ProductMappingProfile()
    {
        CreateMap<Product, ProductListDto>()
            .ForMember(dest => dest.CategoryName, opt => opt.MapFrom(src => src.Category.Name))
            .ForMember(dest => dest.Status, opt => opt.MapFrom(src => src.Status.ToString()));

        CreateMap<Product, ProductDetailsDto>()
            .ForMember(dest => dest.CategoryName, opt => opt.MapFrom(src => src.Category.Name))
            .ForMember(dest => dest.SupplierName, opt => opt.MapFrom(src => src.Supplier != null ? src.Supplier.Name : null))
            .ForMember(dest => dest.Status, opt => opt.MapFrom(src => src.Status.ToString()));

        CreateMap<CreateProductDto, Product>()
            .ForMember(dest => dest.Status, opt => opt.MapFrom(src => ProductStatus.Draft));

        CreateMap<UpdateProductDto, Product>()
            .ForAllMembers(opts => opts.Condition((src, dest, srcMember) => srcMember != null));
    }
}
```

**Service Interface**
```csharp
namespace YourProject.Application.Interfaces;

public interface IProductService
{
    Task<PagedResult<ProductListDto>> GetProductsAsync(ProductFilter filter);
    Task<ProductDetailsDto?> GetProductByIdAsync(long id);
    Task<ProductDetailsDto> CreateProductAsync(CreateProductDto dto);
    Task<ProductDetailsDto> UpdateProductAsync(long id, UpdateProductDto dto);
    Task<bool> DeleteProductAsync(long id);
    Task<List<ProductListDto>> GetLowStockProductsAsync(int threshold);
}
```

**Service Implementation**
```csharp
namespace YourProject.Application.Services;

public class ProductService : IProductService
{
    private readonly IProductRepository _productRepository;
    private readonly IMapper _mapper;
    private readonly ILogger<ProductService> _logger;

    public ProductService(
        IProductRepository productRepository,
        IMapper mapper,
        ILogger<ProductService> logger)
    {
        _productRepository = productRepository;
        _mapper = mapper;
        _logger = logger;
    }

    public async Task<PagedResult<ProductListDto>> GetProductsAsync(ProductFilter filter)
    {
        var query = _productRepository.GetActiveProducts(filter)
            .Include(p => p.Category)
            .OrderByDescending(p => p.CreatedAt);

        return await query
            .ProjectTo<ProductListDto>(_mapper.ConfigurationProvider)
            .ToPagedResultAsync(filter.PageNumber, filter.PageSize);
    }

    public async Task<ProductDetailsDto?> GetProductByIdAsync(long id)
    {
        var product = await _productRepository
            .GetAll()
            .Include(p => p.Category)
            .Include(p => p.Supplier)
            .FirstOrDefaultAsync(p => p.Id == id);

        if (product == null)
        {
            _logger.LogWarning("Product with ID {ProductId} not found", id);
            return null;
        }

        return _mapper.Map<ProductDetailsDto>(product);
    }

    public async Task<ProductDetailsDto> CreateProductAsync(CreateProductDto dto)
    {
        var product = _mapper.Map<Product>(dto);

        await _productRepository.AddAsync(product);

        _logger.LogInformation("Product created with ID {ProductId}", product.Id);

        return _mapper.Map<ProductDetailsDto>(product);
    }

    public async Task<ProductDetailsDto> UpdateProductAsync(long id, UpdateProductDto dto)
    {
        var product = await _productRepository.GetByIdAsync(id);

        if (product == null)
        {
            throw new NotFoundException($"Product with ID {id} not found");
        }

        _mapper.Map(dto, product);
        await _productRepository.UpdateAsync(product);

        _logger.LogInformation("Product {ProductId} updated", id);

        return _mapper.Map<ProductDetailsDto>(product);
    }

    public async Task<bool> DeleteProductAsync(long id)
    {
        var product = await _productRepository.GetByIdAsync(id);

        if (product == null)
        {
            return false;
        }

        await _productRepository.DeleteAsync(id); // Soft delete

        _logger.LogInformation("Product {ProductId} deleted", id);

        return true;
    }

    public async Task<List<ProductListDto>> GetLowStockProductsAsync(int threshold)
    {
        var products = await _productRepository.GetLowStockProductsAsync(threshold);
        return _mapper.Map<List<ProductListDto>>(products);
    }
}
```

### Presentation Layer (API)

**ProductController.cs**
```csharp
namespace YourProject.API.Controllers.V1;

[ApiController]
[Route("api/v{version:apiVersion}/[controller]")]
[ApiVersion("1.0")]
[Authorize]
public class ProductController : ControllerBase
{
    private readonly IProductService _productService;
    private readonly ILogger<ProductController> _logger;

    public ProductController(
        IProductService productService,
        ILogger<ProductController> logger)
    {
        _productService = productService;
        _logger = logger;
    }

    /// <summary>
    /// Get paginated list of products
    /// </summary>
    [HttpGet]
    [OutputCache(PolicyName = "ProductList")]
    [ProducesResponseType(typeof(ApiResponse<PagedResult<ProductListDto>>), StatusCodes.Status200OK)]
    public async Task<ActionResult<ApiResponse<PagedResult<ProductListDto>>>> GetProducts([FromQuery] ProductFilter filter)
    {
        try
        {
            var result = await _productService.GetProductsAsync(filter);
            return Ok(ApiResponse<PagedResult<ProductListDto>>.SuccessResult(result, "Products retrieved successfully"));
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error retrieving products");
            return StatusCode(500, ApiResponse<PagedResult<ProductListDto>>.ErrorResult("An error occurred while retrieving products"));
        }
    }

    /// <summary>
    /// Get product by ID
    /// </summary>
    [HttpGet("{id}")]
    [OutputCache(PolicyName = "ProductDetails")]
    [ProducesResponseType(typeof(ApiResponse<ProductDetailsDto>), StatusCodes.Status200OK)]
    [ProducesResponseType(StatusCodes.Status404NotFound)]
    public async Task<ActionResult<ApiResponse<ProductDetailsDto>>> GetProductById(long id)
    {
        try
        {
            var product = await _productService.GetProductByIdAsync(id);

            if (product == null)
            {
                return NotFound(ApiResponse<ProductDetailsDto>.ErrorResult($"Product with ID {id} not found"));
            }

            return Ok(ApiResponse<ProductDetailsDto>.SuccessResult(product, "Product retrieved successfully"));
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error retrieving product {ProductId}", id);
            return StatusCode(500, ApiResponse<ProductDetailsDto>.ErrorResult("An error occurred while retrieving the product"));
        }
    }

    /// <summary>
    /// Create a new product
    /// </summary>
    [HttpPost]
    [Authorize(Policy = "CanCreateProducts")]
    [ProducesResponseType(typeof(ApiResponse<ProductDetailsDto>), StatusCodes.Status201Created)]
    [ProducesResponseType(StatusCodes.Status400BadRequest)]
    public async Task<ActionResult<ApiResponse<ProductDetailsDto>>> CreateProduct([FromBody] CreateProductDto dto)
    {
        try
        {
            if (!ModelState.IsValid)
            {
                return BadRequest(ApiResponse<ProductDetailsDto>.ErrorResult(
                    "Validation failed",
                    ModelState.Values.SelectMany(v => v.Errors).Select(e => e.ErrorMessage).ToList()
                ));
            }

            var product = await _productService.CreateProductAsync(dto);

            return CreatedAtAction(
                nameof(GetProductById),
                new { id = product.Id },
                ApiResponse<ProductDetailsDto>.SuccessResult(product, "Product created successfully")
            );
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error creating product");
            return StatusCode(500, ApiResponse<ProductDetailsDto>.ErrorResult("An error occurred while creating the product"));
        }
    }

    /// <summary>
    /// Update existing product
    /// </summary>
    [HttpPut("{id}")]
    [Authorize(Policy = "CanEditProducts")]
    [ProducesResponseType(typeof(ApiResponse<ProductDetailsDto>), StatusCodes.Status200OK)]
    [ProducesResponseType(StatusCodes.Status404NotFound)]
    public async Task<ActionResult<ApiResponse<ProductDetailsDto>>> UpdateProduct(long id, [FromBody] UpdateProductDto dto)
    {
        try
        {
            var product = await _productService.UpdateProductAsync(id, dto);
            return Ok(ApiResponse<ProductDetailsDto>.SuccessResult(product, "Product updated successfully"));
        }
        catch (NotFoundException ex)
        {
            return NotFound(ApiResponse<ProductDetailsDto>.ErrorResult(ex.Message));
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error updating product {ProductId}", id);
            return StatusCode(500, ApiResponse<ProductDetailsDto>.ErrorResult("An error occurred while updating the product"));
        }
    }

    /// <summary>
    /// Delete product (soft delete)
    /// </summary>
    [HttpDelete("{id}")]
    [Authorize(Policy = "CanDeleteProducts")]
    [ProducesResponseType(StatusCodes.Status204NoContent)]
    [ProducesResponseType(StatusCodes.Status404NotFound)]
    public async Task<IActionResult> DeleteProduct(long id)
    {
        try
        {
            var result = await _productService.DeleteProductAsync(id);

            if (!result)
            {
                return NotFound(ApiResponse<object>.ErrorResult($"Product with ID {id} not found"));
            }

            return NoContent();
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error deleting product {ProductId}", id);
            return StatusCode(500, ApiResponse<object>.ErrorResult("An error occurred while deleting the product"));
        }
    }

    /// <summary>
    /// Get low stock products
    /// </summary>
    [HttpGet("low-stock")]
    [Authorize(Roles = "Admin,Manager")]
    [ProducesResponseType(typeof(ApiResponse<List<ProductListDto>>), StatusCodes.Status200OK)]
    public async Task<ActionResult<ApiResponse<List<ProductListDto>>>> GetLowStockProducts([FromQuery] int threshold = 10)
    {
        try
        {
            var products = await _productService.GetLowStockProductsAsync(threshold);
            return Ok(ApiResponse<List<ProductListDto>>.SuccessResult(products, $"Found {products.Count} low stock products"));
        }
        catch (Exception ex)
        {
            _logger.LogError(ex, "Error retrieving low stock products");
            return StatusCode(500, ApiResponse<List<ProductListDto>>.ErrorResult("An error occurred while retrieving low stock products"));
        }
    }
}
```

### Testing

**ProductServiceTests.cs**
```csharp
namespace YourProject.Tests.Application.Services;

public class ProductServiceTests
{
    private readonly Mock<IProductRepository> _mockRepository;
    private readonly Mock<IMapper> _mockMapper;
    private readonly Mock<ILogger<ProductService>> _mockLogger;
    private readonly ProductService _service;

    public ProductServiceTests()
    {
        _mockRepository = new Mock<IProductRepository>();
        _mockMapper = new Mock<IMapper>();
        _mockLogger = new Mock<ILogger<ProductService>>();
        _service = new ProductService(_mockRepository.Object, _mockMapper.Object, _mockLogger.Object);
    }

    [Fact]
    public async Task GetProductByIdAsync_ReturnsProduct_WhenProductExists()
    {
        // Arrange
        var product = new Product { Id = 1, Name = "Test Product" };
        var productDto = new ProductDetailsDto { Id = 1, Name = "Test Product" };

        _mockRepository.Setup(r => r.GetAll())
            .Returns(new List<Product> { product }.AsQueryable());

        _mockMapper.Setup(m => m.Map<ProductDetailsDto>(It.IsAny<Product>()))
            .Returns(productDto);

        // Act
        var result = await _service.GetProductByIdAsync(1);

        // Assert
        Assert.NotNull(result);
        Assert.Equal(1, result.Id);
        Assert.Equal("Test Product", result.Name);
    }

    [Fact]
    public async Task CreateProductAsync_CreatesProduct_WhenValid()
    {
        // Arrange
        var createDto = new CreateProductDto { Name = "New Product", Price = 99.99m };
        var product = new Product { Id = 1, Name = "New Product", Price = 99.99m };
        var productDto = new ProductDetailsDto { Id = 1, Name = "New Product", Price = 99.99m };

        _mockMapper.Setup(m => m.Map<Product>(createDto)).Returns(product);
        _mockMapper.Setup(m => m.Map<ProductDetailsDto>(product)).Returns(productDto);
        _mockRepository.Setup(r => r.AddAsync(It.IsAny<Product>())).ReturnsAsync(product);

        // Act
        var result = await _service.CreateProductAsync(createDto);

        // Assert
        Assert.NotNull(result);
        Assert.Equal("New Product", result.Name);
        _mockRepository.Verify(r => r.AddAsync(It.IsAny<Product>()), Times.Once);
    }
}
```

## Key Takeaways

1. **Clean Architecture**: Clear separation of concerns across layers
2. **Repository Pattern**: Data access abstracted from business logic
3. **Service Layer**: Business logic orchestration
4. **DTOs**: Input/output contracts separate from domain entities
5. **AutoMapper**: Automated entity-DTO mapping
6. **Validation**: FluentValidation for complex rules
7. **Error Handling**: Consistent error responses
8. **Testing**: Isolated unit tests with mocks

## See Also

- [architecture-overview.md](architecture-overview.md) - Architecture principles
- [services-and-repositories.md](services-and-repositories.md) - Service and repository patterns
- [validation-patterns.md](validation-patterns.md) - Validation strategies
- [testing-guide.md](testing-guide.md) - Testing approaches
