# Architecture Overview

## Table of Contents
- [Clean Architecture - 4-Layer Pattern](#clean-architecture---4-layer-pattern)
- [Dependency Rule](#dependency-rule)
- [Request Flow](#request-flow)
- [Layer Responsibilities](#layer-responsibilities)
  - [Domain Layer](#1-domain-layer-makatebcore)
  - [Infrastructure Layer](#2-infrastructure-layer-makatebinfrastructure)
  - [Application Layer](#3-application-layer-makatebapplication)
  - [Presentation Layer](#4-presentation-layer-makatebapi)
- [Design Patterns](#design-patterns)
  - [Repository Pattern](#1-repository-pattern)
  - [Service Pattern](#2-service-pattern)
  - [CQRS-lite with IQueryable](#3-cqrs-lite-with-iqueryable)
  - [Soft Delete Pattern](#4-soft-delete-pattern)
  - [Audit Trail Pattern](#5-audit-trail-pattern)
- [Benefits of This Architecture](#benefits-of-this-architecture)
- [Common Mistakes to Avoid](#common-mistakes-to-avoid)

---

## Clean Architecture - 4-Layer Pattern

The Makateb project implements **Clean Architecture** with strict dependency rules and separation of concerns.

### Layer Structure

```
┌─────────────────────────────────────────────────────────┐
│                    Presentation Layer                    │
│                     (Makateb.API)                       │
│                                                          │
│  Controllers, Middleware, Extensions, Program.cs        │
└────────────────────┬────────────────────────────────────┘
                     │ depends on ↓
┌─────────────────────────────────────────────────────────┐
│                   Application Layer                      │
│                 (Makateb.Application)                   │
│                                                          │
│  Services, DTOs, Mapping, EventHandlers                 │
└────────────────────┬────────────────────────────────────┘
                     │ depends on ↓
┌─────────────────────────────────────────────────────────┐
│                 Infrastructure Layer                     │
│                (Makateb.Infrastructure)                 │
│                                                          │
│  Repositories, DbContext, Migrations, External Services │
└────────────────────┬────────────────────────────────────┘
                     │ depends on ↓
┌─────────────────────────────────────────────────────────┐
│                     Domain Layer                         │
│                    (Makateb.Core)                       │
│                                                          │
│  Entities, Interfaces, Enums, Models, Extensions        │
└─────────────────────────────────────────────────────────┘
```

## Dependency Rule

**CRITICAL**: Dependencies must ONLY point inward:

- ✅ API → Application → Infrastructure → Core
- ❌ Core NEVER depends on Infrastructure
- ❌ Application NEVER depends on API
- ❌ Infrastructure NEVER depends on Application

## Request Flow

```
HTTP Request
    ↓
Controller (API Layer)
    │ validates JWT
    │ binds request model
    ↓
Service (Application Layer)
    │ business logic
    │ orchestrates operations
    │ validates business rules
    ↓
Repository (Infrastructure Layer)
    │ data access
    │ EF Core queries
    ↓
DbContext (Infrastructure Layer)
    │ database operations
    ↓
Database (SQL Server)
```

## Layer Responsibilities

### 1. Domain Layer (Makateb.Core)

**Purpose**: Pure business logic with NO external dependencies

**Contains**:
- `Entities/` - Domain entities (Book, ApprovalStep, BookRecipient)
- `Interfaces/` - Repository contracts (IBookRepository, IRepository<T>)
- `Enums/` - Business enums (BookState, SecurityLevel, ImportanceLevel)
- `Models/` - Domain models (Filters, API response wrappers)
- `Extensions/` - Query extensions (IQueryableExtensions)
- `Events/` - Domain events (if using MediatR)

**Key Files**:
- `BaseEntity.cs` - Base class with Id, CreatedAt, UpdatedAt
- `AuditedEntity.cs` - Extends BaseEntity with CreatedBy
- `ISoftDeletable.cs` - Interface for soft delete pattern

**Rules**:
- ❌ NO EF Core usage
- ❌ NO HTTP dependencies
- ❌ NO external service calls
- ✅ ONLY pure C# and domain logic

### 2. Infrastructure Layer (Makateb.Infrastructure)

**Purpose**: Implementation of data access and external services

**Contains**:
- `Data/MakatebDbContext.cs` - EF Core context
- `Repositories/` - Repository implementations
- `Migrations/` - EF Core migrations
- `Services/` - External service integrations (HRService)
- `Helpers/` - Infrastructure utilities

**Implements**:
- Repository interfaces from Core layer
- Database query logic
- External API calls
- File storage operations

**Rules**:
- ✅ Implements interfaces from Core
- ✅ Uses EF Core directly
- ❌ NO business logic (that belongs in Application)

### 3. Application Layer (Makateb.Application)

**Purpose**: Business logic orchestration

**Contains**:
- `Services/` - Business logic (BookService, ApprovalStepService)
- `Interfaces/` - Service contracts (IBooksService)
- `DTOs/` - Data transfer objects (CreateBookDto, BookDetailsDto)
- `Mapping/` - AutoMapper profiles (BookMappingProfile)
- `EventHandlers/` - MediatR event handlers
- `Models/` - Application-specific models

**Responsibilities**:
- Coordinate multiple repositories
- Apply business rules
- Handle domain events
- Transform entities to DTOs

**Rules**:
- ✅ Uses repository interfaces (NOT DbContext directly)
- ✅ Contains ALL business logic
- ❌ NO HTTP concerns (responses, status codes)

### 4. Presentation Layer (Makateb.API)

**Purpose**: HTTP interface and dependency injection configuration

**Contains**:
- `Controllers/V1/` - API controllers
- `Middleware/` - Custom middleware
- `Extensions/` - Configuration extensions
- `Utils/` - API utilities (caching policies, permissions)
- `Models/` - API-specific models (ApiResponse)
- `Program.cs` - Application entry point
- `ProgramConfig.cs` - DI registration extensions

**Responsibilities**:
- HTTP routing
- Request/response binding
- Authentication/Authorization
- Dependency injection configuration
- Middleware pipeline

**Rules**:
- ✅ Uses service interfaces (NOT repositories)
- ✅ Returns HTTP status codes
- ❌ NO business logic
- ❌ NO database access

## Design Patterns

### 1. Repository Pattern

```csharp
// Generic base interface (Core layer)
public interface IRepository<T> where T : class
{
    Task<T?> GetByIdAsync(long id);
    IQueryable<T> GetAll();
    Task<T> AddAsync(T entity);
    Task UpdateAsync(T entity);
    Task DeleteAsync(long id);
}

// Specialized interface (Core layer)
public interface IBookRepository : IRepository<Book>
{
    IQueryable<Book> GetInboxBooks(long unitId, BookFilter filter);
    Task<Book?> GetBookWithDetailsAsync(long id);
}

// Implementation (Infrastructure layer)
public class BookRepository : Repository<Book>, IBookRepository
{
    public BookRepository(MakatebDbContext context) : base(context) { }

    // Custom queries using IQueryable for chainability
}
```

### 2. Service Pattern

```csharp
// Service interface (Application layer)
public interface IBooksService
{
    Task<BookDetailsDto?> GetBookDetailsAsync(long id);
    Task<bool> CreateBookAsync(CreateBookDto dto, bool sendToDraft);
}

// Service implementation (Application layer)
public class BookService : IBooksService
{
    private readonly IBookRepository _bookRepository;
    private readonly IMapper _mapper;

    public BookService(IBookRepository bookRepository, IMapper mapper)
    {
        _bookRepository = bookRepository;
        _mapper = mapper;
    }

    // Heavy business logic here
}
```

### 3. CQRS-lite with IQueryable

Instead of full CQRS, the project uses **IQueryable chaining** for flexible queries:

```csharp
// Repository returns IQueryable for chaining
var query = _bookRepository.GetInboxBooks(unitId, filter);

// Service can further refine
var pagedBooks = await query
    .Include(b => b.Category)
    .ProjectTo<BookListDto>(_mapper.ConfigurationProvider)
    .ToPagedResultAsync(filter.PageNumber, filter.PageSize);
```

### 4. Soft Delete Pattern

All entities implementing `ISoftDeletable` are automatically filtered:

```csharp
// Global query filter in DbContext
modelBuilder.Entity<Book>().HasQueryFilter(e => !e.IsDeleted);

// Soft delete in repository
public override async Task DeleteAsync(long id)
{
    var entity = await GetByIdAsync(id);
    if (entity is ISoftDeletable softDeletable)
    {
        softDeletable.IsDeleted = true;
        await UpdateAsync(entity);
    }
}
```

### 5. Audit Trail Pattern

Base classes automatically track creation and updates:

```csharp
public abstract class AuditedEntity : BaseEntity
{
    public DateTime CreatedAt { get; set; }
    public DateTime? UpdatedAt { get; set; }
    public long? CreatedBy { get; set; }
}

// Set automatically in DbContext.SaveChangesAsync
public override async Task<int> SaveChangesAsync(CancellationToken cancellationToken = default)
{
    foreach (var entry in ChangeTracker.Entries<AuditedEntity>())
    {
        if (entry.State == EntityState.Added)
        {
            entry.Entity.CreatedAt = DateTime.UtcNow;
            entry.Entity.CreatedBy = GetCurrentUserId();
        }
        else if (entry.State == EntityState.Modified)
        {
            entry.Entity.UpdatedAt = DateTime.UtcNow;
        }
    }

    return await base.SaveChangesAsync(cancellationToken);
}
```

## Benefits of This Architecture

1. **Testability**: Each layer can be tested independently
2. **Maintainability**: Clear separation of concerns
3. **Flexibility**: Can swap infrastructure without touching business logic
4. **Scalability**: Easy to add new features following established patterns
5. **Clean Code**: SOLID principles enforced by layer boundaries

## Common Mistakes to Avoid

❌ **Using DbContext directly in controllers**
```csharp
// WRONG
public class BookController : ControllerBase
{
    private readonly MakatebDbContext _context; // ❌ NO!
}
```

❌ **Business logic in controllers**
```csharp
// WRONG
[HttpPost]
public async Task<IActionResult> CreateBook(CreateBookDto dto)
{
    // 100 lines of business logic ❌ NO!
}
```

❌ **Repositories calling other repositories**
```csharp
// WRONG
public class BookRepository : Repository<Book>
{
    private readonly IApprovalStepRepository _approvalRepo; // ❌ NO!
}
```

❌ **Infrastructure layer depending on Application layer**
```csharp
// WRONG - Infrastructure project referencing Application
using Makateb.Application.Services; // ❌ NO!
```

## See Also

- [controllers-and-routing.md](controllers-and-routing.md) - Controller patterns
- [services-and-repositories.md](services-and-repositories.md) - Service and repository implementation
- [entity-framework-patterns.md](entity-framework-patterns.md) - EF Core and DbContext patterns
- [complete-examples.md](complete-examples.md) - Full feature examples
