---
name: angular-dev-guidelines
description: Frontend development guidelines for Angular 19/TypeScript applications. Modern patterns including standalone components, signals, RxJS state management, file organization with features directory, Bootstrap styling, performance optimization, and TypeScript best practices. Use when creating components, services, features, fetching data, styling, routing, or working with frontend code.
---

# Angular 19 Development Guidelines

> **⚠️ MANDATORY: All patterns and guidelines in this skill MUST be followed strictly. These are not suggestions - they are required standards.**

## Purpose

Comprehensive guide for modern Angular 19 development, emphasizing standalone components, signals, RxJS-based state management, proper file organization, and performance optimization.

## When to Use This Skill

Automatically activates when working on:
- Creating new components or pages
- Building new features
- Implementing services and state management
- Setting up routing with Angular Router
- Styling components with Bootstrap
- Performance optimization
- Organizing frontend code
- TypeScript best practices
- RxJS patterns and operators

## ⚠️ CRITICAL: How to Use This Skill Properly

When this skill activates:
1. **Read this main file first** for overview and core principles
2. **Check the Navigation Guide table below** - it tells you which resource file to read based on your task
3. **ALWAYS read the recommended resource file** - it contains detailed patterns and examples
4. **Apply the patterns from BOTH files** - main file + resource file

**Example:**
- Task: "Create a user list component that fetches data from API"
- Step 1: ✅ Read this main skill.md (you're doing this)
- Step 2: ✅ Check Navigation Guide → "Component with data fetching" → component-patterns.md + data-fetching.md
- Step 3: ✅ **READ both resource files** before writing code
- Step 4: ✅ Apply component patterns and data fetching patterns from those resources

---

## Quick Start

### New Component Checklist

Creating a component? Follow this checklist:

- [ ] Use standalone component pattern
- [ ] Define TypeScript interface for component inputs
- [ ] Use signals for reactive state management
- [ ] Lazy load feature modules when appropriate
- [ ] Use Bootstrap classes for styling
- [ ] Implement OnDestroy for cleanup
- [ ] Use async pipe for observables in templates
- [ ] Follow Angular style guide naming conventions
- [ ] Use dependency injection properly
- [ ] Implement change detection strategy OnPush when possible

### New Feature Checklist

Creating a feature? Set up this structure:

- [ ] Create `src/app/features/{feature-name}/` directory
- [ ] Create subdirectories: `services/`, `components/`, `models/`, `guards/`
- [ ] Create feature service: `services/{feature}.service.ts`
- [ ] Set up TypeScript interfaces in `models/`
- [ ] Create routing module if feature has routes
- [ ] Use standalone components
- [ ] Implement proper error handling
- [ ] Export public API from feature module/component

---

## Common Imports Cheatsheet

```typescript
// Angular Core
import { Component, OnInit, OnDestroy, inject, signal } from '@angular/core';
import { CommonModule } from '@angular/common';
import { FormsModule, ReactiveFormsModule } from '@angular/forms';

// Angular Router
import { Router, ActivatedRoute } from '@angular/router';

// RxJS
import { Observable, Subject, BehaviorSubject } from 'rxjs';
import { takeUntil, map, filter, switchMap, tap } from 'rxjs/operators';

// HTTP Client
import { HttpClient, HttpErrorResponse } from '@angular/common/http';

// Services
import { MyFeatureService } from './services/my-feature.service';

// Models
import { User, UserRole } from './models/user.model';
```

---

## Navigation Guide (⚠️ MANDATORY - Check This Table and Read Relevant Resources)

Use this table to determine which resource file you MUST read based on your task:

| If You Need To... | Then MUST Read... | When to Read It |
|-------------------|-------------------|-----------------|
| **Create components, pages, lifecycle hooks** | **[component-patterns.md](resources/component-patterns.md)** | **When creating ANY component** |
| **Fetch data from API, HTTP calls, state management** | **[data-fetching.md](resources/data-fetching.md)** | **⚡ CRITICAL: When making HTTP requests** |
| Organize project structure, features, folders | [file-organization.md](resources/file-organization.md) | Before starting new features |
| Style components, use Bootstrap utilities | [styling-guide.md](resources/styling-guide.md) | When implementing UI/styling |
| Set up routing, guards, lazy loading | [routing-guide.md](resources/routing-guide.md) | When configuring routes |
| Handle loading states, errors, async data | [loading-and-error-states.md](resources/loading-and-error-states.md) | When implementing data loading |
| Optimize performance, OnPush, trackBy | [performance.md](resources/performance.md) | When optimizing components |
| TypeScript types, interfaces, strict mode | [typescript-standards.md](resources/typescript-standards.md) | When defining models/types |
| Forms, interceptors, guards, pipes | [common-patterns.md](resources/common-patterns.md) | When implementing common features |

**⚡ Most Common Mistakes:**
1. Creating components without reading component-patterns.md (missing OnPush, proper lifecycle, signals)
2. Making HTTP calls without reading data-fetching.md (missing error handling, memory leaks from unsubscribed observables)
3. Not using async pipe or proper cleanup (memory leaks)
4. ⭐ **For styling questions (badges, status indicators, UI patterns), ALWAYS check common-patterns.md FIRST** before searching codebase - ensures consistency across all pages

---

## Topic Guides (Quick Reference - Always Check Navigation Guide Above)

### 🎨 Component Patterns
Modern Angular 19 standalone components with signals, OnPush, and proper lifecycle.
**[📖 Complete Guide: resources/component-patterns.md](resources/component-patterns.md)**

### 📊 Data Fetching & State Management
Services, HttpClient, RxJS observables, signals, error handling, and state patterns.
**[📖 Complete Guide: resources/data-fetching.md](resources/data-fetching.md)**

### 📁 File Organization
Feature-based structure, services/, components/, models/, guards/ organization.
**[📖 Complete Guide: resources/file-organization.md](resources/file-organization.md)**

### 🎨 Styling with Bootstrap
Bootstrap 5 utilities, component styles, scoped styling, ViewEncapsulation.
**[📖 Complete Guide: resources/styling-guide.md](resources/styling-guide.md)**

### 🛣️ Routing
Standalone routes, lazy loading, guards, resolvers, route configuration.
**[📖 Complete Guide: resources/routing-guide.md](resources/routing-guide.md)**

### ⏳ Loading & Error States
Async pipe, loading indicators, error handling, user feedback patterns.
**[📖 Complete Guide: resources/loading-and-error-states.md](resources/loading-and-error-states.md)**

### ⚡ Performance
OnPush strategy, lazy loading, trackBy, pure pipes, signals, unsubscribe patterns.
**[📖 Complete Guide: resources/performance.md](resources/performance.md)**

### 📘 TypeScript Standards
Strict mode, explicit types, interfaces, no `any`, null handling.
**[📖 Complete Guide: resources/typescript-standards.md](resources/typescript-standards.md)**

### 🔧 Common Patterns
Reactive forms, interceptors, guards, directives, pipes, RxJS operators.
**[📖 Complete Guide: resources/common-patterns.md](resources/common-patterns.md)**

---

## Core Principles

1. **Standalone Components**: Use standalone components, no NgModule needed
2. **Signals First**: Use signals for reactive state management
3. **OnPush Change Detection**: Optimize performance with OnPush strategy
4. **RxJS for Async**: Use observables for async operations
5. **Features are Organized**: services/, components/, models/ subdirs
6. **Bootstrap for Styling**: Utility-first with Bootstrap classes
7. **Lazy Load Features**: Improve initial load performance
8. **Type Safety**: Strict TypeScript, no `any` types
9. **Proper Cleanup**: Unsubscribe from observables in ngOnDestroy

---

## Quick Reference: File Structure

```
src/app/
  features/
    users/
      services/
        user.service.ts
      components/
        user-header.component.ts
      pages/
        user-list.component.ts
        user-detail.component.ts
      models/
        user.model.ts
      guards/
        user.guard.ts
      users.routes.ts

  shared/
    components/
      loading-spinner/
      error-message/
    directives/
      highlight.directive.ts
    pipes/
      custom-date.pipe.ts

  core/
    services/
      auth.service.ts
      api.service.ts
    interceptors/
      auth.interceptor.ts
      error.interceptor.ts
```

---

## Modern Component Template (Quick Copy)

```typescript
import { Component, OnInit, OnDestroy, inject, signal } from '@angular/core';
import { CommonModule } from '@angular/common';
import { Subject, takeUntil } from 'rxjs';
import { MyFeatureService } from '../services/my-feature.service';
import { MyData } from '../models/my-data.model';

@Component({
  selector: 'app-my-component',
  standalone: true,
  imports: [CommonModule],
  templateUrl: './my-component.component.html',
  styleUrls: ['./my-component.component.scss']
})
export class MyComponent implements OnInit, OnDestroy {
  private myService = inject(MyFeatureService);
  private destroy$ = new Subject<void>();

  // Signals for reactive state
  data = signal<MyData[]>([]);
  isLoading = signal(false);
  error = signal<string | null>(null);

  ngOnInit(): void {
    this.loadData();
  }

  loadData(): void {
    this.isLoading.set(true);
    this.myService.getData()
      .pipe(takeUntil(this.destroy$))
      .subscribe({
        next: (data) => {
          this.data.set(data);
          this.isLoading.set(false);
        },
        error: (err) => {
          this.error.set(err.message);
          this.isLoading.set(false);
        }
      });
  }

  ngOnDestroy(): void {
    this.destroy$.next();
    this.destroy$.complete();
  }
}
```

---

## Anti-Patterns to Avoid

❌ Creating components without OnPush change detection strategy
❌ Not unsubscribing from observables (memory leaks!)
❌ Using `.subscribe()` in templates (use async pipe instead)
❌ Not handling error states in HTTP calls
❌ Using `any` type instead of proper interfaces
❌ Missing loading indicators for async operations
❌ Not using trackBy in *ngFor loops
❌ Forgetting to implement ngOnDestroy for cleanup
❌ Subscribing multiple times to the same observable
❌ Not using signals for local reactive state
❌ Putting business logic in components instead of services

---

## Related Skills

- **dotnet-backend-guidelines**: .NET Core backend API patterns that Angular consumes

---

**Skill Status**: Adapted for Angular 19 + Bootstrap
