# Performance Optimization

## Table of Contents
- [OnPush Change Detection](#onpush-change-detection)
- [Lazy Loading](#lazy-loading)
- [TrackBy Functions](#trackby-functions)
- [Pure Pipes](#pure-pipes)
- [Signals for Reactivity](#signals-for-reactivity)
- [Unsubscribe from Observables](#unsubscribe-from-observables)
- [Virtual Scrolling](#virtual-scrolling)
- [Debounce User Input](#debounce-user-input)
- [Optimize Images](#optimize-images)
- [Bundle Optimization](#bundle-optimization)
- [Performance Checklist](#performance-checklist)

---

## OnPush Change Detection

```typescript
import { ChangeDetectionStrategy } from '@angular/core';

@Component({
  selector: 'app-user-list',
  standalone: true,
  changeDetection: ChangeDetectionStrategy.OnPush, // ✅ Use OnPush
  template: `...`
})
export class UserListComponent {
  // Use signals - they work perfectly with OnPush
  users = signal<User[]>([]);
}
```

**Benefits:**
- Only checks when inputs change or events fire
- Massive performance improvement for large component trees
- Works seamlessly with signals

## Lazy Loading

```typescript
// app.routes.ts
export const routes: Routes = [
  {
    path: 'users',
    loadComponent: () => import('./features/users/user-list.component')
      .then(m => m.UserListComponent) // ✅ Lazy load
  }
];
```

**Benefits:**
- Smaller initial bundle
- Faster first page load
- Load features on demand

## TrackBy Functions

```typescript
// ❌ BAD - Recreates DOM on every change
<div *ngFor="let user of users()">{{ user.name }}</div>

// ✅ GOOD - Tracks by ID, only updates changed items
<div *ngFor="let user of users(); trackBy: trackByUserId">
  {{ user.name }}
</div>
```

```typescript
trackByUserId(index: number, user: User): number {
  return user.id;
}
```

## Pure Pipes

```typescript
@Pipe({
  name: 'filterUsers',
  standalone: true,
  pure: true // ✅ Default, only recalculates when inputs change
})
export class FilterUsersPipe implements PipeTransform {
  transform(users: User[], searchTerm: string): User[] {
    return users.filter(u => u.name.includes(searchTerm));
  }
}
```

## Signals for Reactivity

```typescript
// ✅ GOOD - Signals are efficient
export class UserListComponent {
  users = signal<User[]>([]);
  searchTerm = signal('');

  // Computed signal - only recalculates when dependencies change
  filteredUsers = computed(() => {
    const term = this.searchTerm().toLowerCase();
    return this.users().filter(u => u.name.toLowerCase().includes(term));
  });
}
```

## Unsubscribe from Observables

```typescript
import { Subject, takeUntil } from 'rxjs';

export class UserListComponent implements OnDestroy {
  private destroy$ = new Subject<void>();

  ngOnInit() {
    this.userService.getUsers()
      .pipe(takeUntil(this.destroy$)) // ✅ Auto-unsubscribe
      .subscribe(users => this.users.set(users));
  }

  ngOnDestroy() {
    this.destroy$.next();
    this.destroy$.complete();
  }
}

// Or use async pipe - auto-unsubscribes
users$ = this.userService.getUsers(); // ✅ Template: users$ | async
```

## Virtual Scrolling

```typescript
import { ScrollingModule } from '@angular/cdk/scrolling';

@Component({
  imports: [ScrollingModule],
  template: `
    <cdk-virtual-scroll-viewport itemSize="50" class="viewport">
      <div *cdkVirtualFor="let user of users()">
        {{ user.name }}
      </div>
    </cdk-virtual-scroll-viewport>
  `
})
export class UserListComponent {
  users = signal<User[]>([]); // Large list
}
```

## Debounce User Input

```typescript
import { debounceTime, distinctUntilChanged } from 'rxjs/operators';

searchControl = new FormControl('');

ngOnInit() {
  this.searchControl.valueChanges
    .pipe(
      debounceTime(300), // Wait 300ms after user stops typing
      distinctUntilChanged(), // Only if value changed
      takeUntil(this.destroy$)
    )
    .subscribe(term => this.search(term));
}
```

## Optimize Images

```html
<!-- Lazy load images -->
<img [src]="user.avatar" loading="lazy" alt="User avatar">

<!-- Use responsive images -->
<img
  [srcset]="user.avatar + ' 1x, ' + user.avatarHd + ' 2x'"
  [src]="user.avatar"
  alt="User avatar">
```

## Bundle Optimization

```typescript
// ❌ BAD - Imports entire lodash
import _ from 'lodash';

// ✅ GOOD - Import only what you need
import { debounce } from 'lodash-es';
```

## Performance Checklist

✅ **OnPush change detection** for most components
✅ **Lazy load features** via routing
✅ **TrackBy functions** in *ngFor loops
✅ **Pure pipes** for transformations
✅ **Signals** for reactive state
✅ **Unsubscribe** from observables (takeUntil or async pipe)
✅ **Virtual scrolling** for long lists (1000+ items)
✅ **Debounce** user input
✅ **Optimize images** (lazy loading, responsive)
✅ **Tree-shakeable imports** (import specific functions)

❌ **Don't use Default change detection** everywhere
❌ **Don't eager load all features**
❌ **Don't forget trackBy** in large lists
❌ **Don't create impure pipes** unnecessarily
❌ **Don't forget to unsubscribe**
